import { spawn } from "bun";
import Redis from "ioredis";

// 任务执行入口，自动设置 SLACK_WEBHOOK 并依次运行 gsheet、gsheet:ai 和 notify-slack
async function main() {
  const env = {
    ...process.env
  };

  const redis = new Redis(Bun.env.REDIS_URL);
  let task_trigger = "manual";

  const shouldRun = await redis.get("falcon:should_run_tasks");

  if (shouldRun === "true") {
    task_trigger = "should_run_tasks";
    await redis.del("falcon:should_run_tasks");
  } else if (isMonday11AM()) {
    const mondayKey = "falcon:monday_11am_executed";
    const alreadyExecuted = await redis.get(mondayKey);
    if (alreadyExecuted) {
      task_trigger = "cron";
      console.log(`[${task_trigger}] 本周一11点任务已执行，跳过。`);
      await redis.quit();
      process.exit(0);
    } else {
      task_trigger = "cron";
      // 设置标记，24小时后自动过期
      await redis.set(mondayKey, "true", "EX", 60 * 60 * 24);
      console.log(`[${task_trigger}] 定时任务触发，已设置本周标记。`);
    }
  } else {
    task_trigger = "manual";
    console.log(`[${task_trigger}] should_run_tasks 不是 'true'，且不是定时任务，跳过。`);
    await redis.quit();
    process.exit(0);
  }

  // 运行 bun run detect
  console.log(`[${task_trigger}] 开始执行 detect`);
  const detectProcess = spawn({
    cmd: ["bun", "run", "detect"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });
  await detectProcess.exited;
  console.log(`[${task_trigger}] detect 执行完成`);

  // 运行 bun run task:gsheet
  console.log(`[${task_trigger}] 开始执行 task:gsheet`);
  const gsheetProcess = spawn({
    cmd: ["bun", "run", "task:gsheet"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });
  await gsheetProcess.exited;
  console.log(`[${task_trigger}] task:gsheet 执行完成`);

  // 延迟执行 task:gsheet:ai 避免 Google Sheet 限流
  const delayMs = parseInt(process.env.GSHEET_AI_DELAY_MS ?? "60000", 10);
  console.log(`[${task_trigger}] Waiting ${delayMs}ms before running task:gsheet:ai to avoid rate limiting`);
  await new Promise<void>(resolve => setTimeout(resolve, delayMs));

  // 运行 bun run task:gsheet:ai
  console.log(`[${task_trigger}] 开始执行 task:gsheet:ai`);
  const gsheetAiProcess = spawn({
    cmd: ["bun", "run", "task:gsheet:ai"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });
  await gsheetAiProcess.exited;
  console.log(`[${task_trigger}] task:gsheet:ai 执行完成`);

  // 运行 bun run notify-slack
  console.log(`[${task_trigger}] 开始执行 notify-slack`);
  const notifySlackProcess = spawn({
    cmd: ["bun", "run", "notify-slack"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });
  await notifySlackProcess.exited;
  console.log(`[${task_trigger}] notify-slack 执行完成`);

  if (task_trigger === "should_run_tasks") {
    console.log(`[${task_trigger}] should_run_tasks 执行完毕，已清理 redis key。`);
  } else if (task_trigger === "cron") {
    console.log(`[${task_trigger}] cron 任务执行完毕。`);
  } else {
    console.log(`[${task_trigger}] 手动触发任务执行完毕。`);
  }

  await redis.quit();

  console.log(`[${task_trigger}] 任务执行完毕，退出。`);
  process.exit(0);
}

function isMonday11AM() {
  const now = new Date();
  return now.getDay() === 1 && now.getHours() === 11;
}

// 调用入口
main()
  .then()
  .catch(err => {
    console.error(err);
    process.exit(1);
  });