// @ts-nocheck
import { google } from 'googleapis';
import { readdirSync, readFileSync } from 'fs';
import { join } from 'path';

async function uploadAllToSheets() {
  const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';
  const spreadsheetId = useAiSelectors
    ? process.env.GOOGLE_SHEET_AI_ID!
    : process.env.GOOGLE_SHEET_ID!;
  const keyFile = process.env.GOOGLE_SERVICE_ACCOUNT_KEY_FILE!;
  const auth = new google.auth.GoogleAuth({
    keyFile,
    scopes: ['https://www.googleapis.com/auth/spreadsheets'],
  });
  const sheets = google.sheets({ version: 'v4', auth });

  // Determine directories to upload: both diff and nav folders for AI or baseline
  const dirNames = useAiSelectors
    ? ['diffAINavWithBaseline', 'diffAIWithBaseline']
    : ['diffNavWithBaseline', 'diffWithBaseline'];
    
  // 拆分主目录(primary)和备用目录(secondary)。
  // 读取主目录下所有 JSON，检查是否有非空数组。
  // 如果主目录有内容，只上传主目录文件；否则上传备用目录文件。
  const [primary, secondary] = dirNames;
  const primaryDir = join(__dirname, primary);
  const primaryFiles = readdirSync(primaryDir).filter(f => f.endsWith('.json'));
  let usePrimary = false;
  for (const f of primaryFiles) {
    const raw = readFileSync(join(primaryDir, f), 'utf-8');
    const data = JSON.parse(raw);
    if (Array.isArray(data) && data.length > 0) {
      usePrimary = true;
      break;
    }
  }
  const dirsToUpload = usePrimary ? [primary] : [secondary];
  const files = dirsToUpload.flatMap(subdir => {
    const d = join(__dirname, subdir);
    return readdirSync(d)
      .filter(f => f.endsWith('.json'))
      .map(f => ({ subdir, file: f, path: join(d, f) }));
  });

  // Fetch existing sheet names
  const { data: spreadsheet } = await sheets.spreadsheets.get({ spreadsheetId });
  // Prepare and clean up sheets: delete all but the first to avoid empty spreadsheet
  const sheetsArr = spreadsheet.sheets || [];
  // Delete extra sheets (keep index 0)
  if (sheetsArr.length > 1) {
    const deleteRequests = sheetsArr.slice(1).map(s => ({ deleteSheet: { sheetId: s.properties?.sheetId } }));
    await sheets.spreadsheets.batchUpdate({ spreadsheetId, requestBody: { requests: deleteRequests } });
  }
  // Track existing titles (only first sheet remains)
  const existing: string[] = [sheetsArr[0]?.properties?.title || ''];

  // Rename the remaining sheet to match the first file (if any)
  if (files.length > 0) {
    const firstName = files[0].file.replace(/\.json$/, '');
    const firstSheetId = sheetsArr[0].properties?.sheetId!;
    if (existing[0] !== firstName) {
      await sheets.spreadsheets.batchUpdate({
        spreadsheetId,
        requestBody: { requests: [
          { updateSheetProperties: { properties: { sheetId: firstSheetId, title: firstName }, fields: 'title' } }
        ] }
      });
    }
    existing[0] = firstName;
  }

  for (const { subdir, file, path } of files) {
    // Read and skip empty JSON
    const fileRaw = readFileSync(path, 'utf-8');
    const fileData = JSON.parse(fileRaw);
    if (Array.isArray(fileData) && fileData.length === 0) {
      console.log(`🚫 Skipping empty file ${file}`);
      continue;
    }
    // Use file name (without extension) as sheetName
    const sheetName = file.replace(/\.json$/, '');
    if (!existing.includes(sheetName)) {
      // Create new sheet
      await sheets.spreadsheets.batchUpdate({
        spreadsheetId,
        requestBody: {
          requests: [{ addSheet: { properties: { title: sheetName } } }]
        }
      });
      existing.push(sheetName);
    }

    // Always clear existing values, even for empty data, to overwrite the sheet
    await sheets.spreadsheets.values.clear({ spreadsheetId, range: `'${sheetName}'` });
    if (fileData.length) {
      const headers = Object.keys(fileData[0]);
      const rows = fileData.map((row: any) =>
        headers.map((h: string) => {
          const cell = row[h];
          return cell != null && typeof cell === 'object'
            ? JSON.stringify(cell)
            : cell;
        })
      );

      // Write headers and data
      await sheets.spreadsheets.values.append({
        spreadsheetId,
        range: `'${sheetName}'!A1`,
        valueInputOption: 'RAW',
        requestBody: { values: [headers, ...rows] }
      });
    }

    console.log(`✅ Uploaded ${file} to tab ${sheetName}`);
  }
}

async function main() {
  try {
    await uploadAllToSheets();
  } catch (err) {
    console.error('Error in task:gsheet workflow:', err);
    process.exit(1);
  }
}

main();
