import { WebClient } from '@slack/web-api';
import { createReadStream } from 'fs';
import { basename } from 'path';

export async function sendSlackMessage(webhookUrl: string, params: {
  issue: string,
  displayName: string,
  language: string,
  siteEditor: string
}) {
  const text = [
    `Issue: ${params.issue}`,
    `Display Name: ${params.displayName}`,
    `Language: ${params.language}`,
    `Site editor: ${params.siteEditor}`
  ].join('\n');

  const res = await fetch(webhookUrl, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ text })
  });

  if (!res.ok) {
    throw new Error(`Slack 推送失败: ${res.status} ${res.statusText}`);
  }
}

/**
 * Upload a file to Slack.
 * @param token Bot token (e.g., xoxb-...)
 * @param channel Slack channel ID or name (e.g., #general)
 * @param filePath Local path to the file to upload
 * @param initialComment Optional comment to post with the file
 */
export async function sendSlackFile(
  token: string,
  channel: string,
  filePath: string,
  initialComment?: string
) {
  const web = new WebClient(token);
  const response = await web.files.uploadV2({
    channels: channel,
    file: createReadStream(filePath),
    filename: basename(filePath),
    initial_comment: initialComment,
  });

  if (!response.ok) {
    throw new Error(`Slack 文件上传失败: ${response.error}`);
  }
}

// 示例用法
// sendSlackMessage('https://hooks.slack.com/services/T00000000/B00000000/XXXXXXXXXXXXXXXXXXXXXXXX', {
//   issue: 'Theme color missing',
//   displayName: 'Lavender',
//   language: 'en',
//   siteEditor: 'https://www.strikingly.com/s/sites/12572142/edit'
// });