// @ts-nocheck
import { diff } from 'deep-diff';
import { readFileSync, writeFileSync, mkdirSync, rmSync } from 'fs';
import { join } from 'path';
import { section_selectors, sectionSelectorLanguages, ai_section_selectors, aiSectionSelectorsLanguages } from '../constant/section_selectors';

// Directory containing JSON outputs to compare
const srcDir = 'src/detect_section_selector_masters/extractOutput';
// Directory to write diff results
const outDir = 'src/detect_section_selector_masters/diffWithBaseline';

// Remove existing diffWithBaseline folder if it exists to start fresh
rmSync(outDir, { recursive: true, force: true });

// Ensure output directory exists
mkdirSync(outDir, { recursive: true });

const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';
// 根据环境变量选择使用哪个 selector 数组
const selectedSelectors = useAiSelectors ? ai_section_selectors : section_selectors;
const selectedLanguages = useAiSelectors ? aiSectionSelectorsLanguages : sectionSelectorLanguages;

// Baseline section ID (override via BASELINE_ID env variable)
const EN_SITE_ID = 27328697;
const AI_EN_SITE_ID = 29279960;
const baselineId = useAiSelectors ? AI_EN_SITE_ID : EN_SITE_ID;
const baselineFile = `${baselineId}_extract.json`;

// Read baseline data
const baselinePath = join(srcDir, baselineFile);
const baselineData = JSON.parse(readFileSync(baselinePath, 'utf-8'));

// Fields to ignore during diff
const ignoreFields = ['pageUid', 'pageTitle', 'id', 'path'];

selectedSelectors
  .filter(id => Number(id) !== baselineId)
  .forEach(otherId => {
    const otherFile = `${otherId}_extract.json`;
    const otherPath = join(srcDir, otherFile);
    const otherData = JSON.parse(readFileSync(otherPath, 'utf-8'));

    const changes = diff(baselineData, otherData, (path, key) => ignoreFields.includes(key)) || [];

    // Annotate each change with context
    const annotated = changes.map(change => {
      const context: any = {};
      const first = change.path && change.path[0];
      if (typeof first === 'number') {
        const pageA = baselineData[first] || {};
        const pageB = otherData[first] || {};
        // Build flattened context keys with language prefix
        const langA = selectedLanguages[baselineId];
        const langB = selectedLanguages[otherId];
        // pageUid, pageTitle, sectionIndex
        context[`${langA}:subcategory`] = pageA.pageTitle;
        context[`${langB}:subcategory`] = pageB.pageTitle;
        context[`${langA}:sectionIndex`] = pageA.sectionIndex;
        context[`${langB}:sectionIndex`] = pageB.sectionIndex;
        context[`${langA}:pageUid`] = pageA.pageUid;
        context[`${langB}:pageUid`] = pageB.pageUid;
        const idx = change.path.indexOf('result');
        if (idx !== -1) {
          const itemIdx = change.path[idx + 1];
          // Attach id and path with language prefix
          context[`${langA}:id`] = pageA.result?.[itemIdx]?.id || 'Not Applicable';
          context[`${langB}:id`] = pageB.result?.[itemIdx]?.id || 'Not Applicable';
          context[`${langA}:path`] = pageA.result?.[itemIdx]?.path || 'Not Applicable';
          context[`${langB}:path`] = pageB.result?.[itemIdx]?.path || 'Not Applicable';
        }
      }
      return { ...change, context };
    });

    // Convert to semantic format
    const semantic = annotated.map(change => {
      const map: Record<string, string> = { E: '修改', N: '新增', D: '删除', A: '数组类型的变更' };
      const type = map[change.kind] || change.kind;
      const pathStr = (change.path || []).filter(seg => typeof seg !== 'number' && seg !== 'result').join('.');
      let oldV: any, newV: any;
      if (change.kind === 'E') {
        oldV = (change as any).lhs;
        newV = (change as any).rhs;
      } else if (change.kind === 'N') {
        oldV = (change as any).lhs;
        newV = (change as any).rhs;
      } else if (change.kind === 'D') {
        oldV = (change as any).lhs;
        newV = (change as any).rhs;
      } else if (change.kind === 'A') {
        oldV = (change as any).item.lhs;
        newV = (change as any).item.rhs;
      }
      return {
        [`${selectedLanguages[baselineId]} VS ${selectedLanguages[otherId]} 更改类型`]: type,
        "字段类型": pathStr,
        [`${selectedLanguages[baselineId]}:模版值`]: oldV || 'Not Applicable',
        [`${selectedLanguages[otherId]}:模版值`]: newV || 'Not Applicable',
        ...change.context
      };
    });

    const outFile = `${selectedLanguages[baselineId]}_vs_${selectedLanguages[otherId]}_diff.json`;
    const outPath = join(outDir, outFile);
    writeFileSync(outPath, JSON.stringify(semantic, null, 2), 'utf-8');
    console.log(`✅ Diff between ${baselineId} and ${otherId} written to ${join(outDir, outFile)}  ${useAiSelectors ? '(AI selectors)' : ''}`);
  });