// @ts-nocheck
import { google } from 'googleapis';
import { readdirSync, readFileSync } from 'fs';
import { join } from 'path';

async function uploadAllToSheets() {
  const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';
  const spreadsheetId = useAiSelectors
    ? process.env.GOOGLE_SHEET_AI_ID!
    : process.env.GOOGLE_SHEET_ID!;
  const keyFile = process.env.GOOGLE_SERVICE_ACCOUNT_KEY_FILE!;
  const auth = new google.auth.GoogleAuth({
    keyFile,
    scopes: ['https://www.googleapis.com/auth/spreadsheets'],
  });
  const sheets = google.sheets({ version: 'v4', auth });

  // Determine directories to upload: both diff and nav folders for AI or baseline
  const dirNames = useAiSelectors
    ? ['diffAINavWithBaseline', 'diffAIWithBaseline']
    : ['diffNavWithBaseline', 'diffWithBaseline'];
    
  // 拆分主目录(primary)和备用目录(secondary)。
  // 读取主目录下所有 JSON，检查是否有非空数组。
  // 如果主目录有内容，只上传主目录文件；否则上传备用目录文件。
  const [primary, secondary] = dirNames;
  const primaryDir = join(__dirname, primary);
  const primaryFiles = readdirSync(primaryDir).filter(f => f.endsWith('.json'));
  let usePrimary = false;
  for (const f of primaryFiles) {
    const raw = readFileSync(join(primaryDir, f), 'utf-8');
    const data = JSON.parse(raw);
    if (Array.isArray(data) && data.length > 0) {
      usePrimary = true;
      break;
    }
  }
  const dirsToUpload = usePrimary ? [primary] : [secondary];
  const files = dirsToUpload.flatMap(subdir => {
    const d = join(__dirname, subdir);
    return readdirSync(d)
      .filter(f => f.endsWith('.json'))
      .map(f => ({ subdir, file: f, path: join(d, f) }));
  });

  // Fetch existing sheet names
  const { data: spreadsheet } = await sheets.spreadsheets.get({ spreadsheetId });
  const existing = spreadsheet.sheets?.map((s: any) => s.properties?.title || '') || [];

  for (const { subdir, file, path } of files) {
    // Prefix sheet name with directory to distinguish diff vs nav
    const sheetName = `${subdir}-${file.replace(/\.json$/, '')}`;
    if (!existing.includes(sheetName)) {
      // Create new sheet
      await sheets.spreadsheets.batchUpdate({
        spreadsheetId,
        requestBody: {
          requests: [{ addSheet: { properties: { title: sheetName } } }]
        }
      });
    }

    const raw = readFileSync(path, 'utf-8');
    const data = JSON.parse(raw);
    // Always clear existing values, even for empty data, to overwrite the sheet
    await sheets.spreadsheets.values.clear({ spreadsheetId, range: `'${sheetName}'` });
    if (data.length) {
      const headers = Object.keys(data[0]);
      const rows = data.map((row: any) =>
        headers.map((h: string) => {
          const cell = row[h];
          return cell != null && typeof cell === 'object'
            ? JSON.stringify(cell)
            : cell;
        })
      );

      // Write headers and data
      await sheets.spreadsheets.values.append({
        spreadsheetId,
        range: `'${sheetName}'!A1`,
        valueInputOption: 'RAW',
        requestBody: { values: [headers, ...rows] }
      });
    }

    console.log(`✅ Uploaded ${file} to tab ${sheetName}`);
  }
}

async function main() {
  try {
    await uploadAllToSheets();
  } catch (err) {
    console.error('Error in task:gsheet workflow:', err);
    process.exit(1);
  }
}

main();
