// @ts-nocheck
import { JSDOM } from "jsdom";

export const getNavigationInfo = (items, pathArr: string[] = []) => {
  let navigationResult = [];

  items?.forEach((item, idx) => {
    // 构建对象路径
    const currentPathArr = [...pathArr, `items[${idx}]`];
    const type = item.type;
    const itemsCount = item.items ? item.items.length : 0;

    navigationResult.push({
      patch: currentPathArr.join("."),
      type: type,
      itemsCount: itemsCount,
    });

    // 递归处理 dropdown
    if (item.type === "dropdown" && item.items) {
      navigationResult = navigationResult.concat(
        getNavigationInfo(item.items, currentPathArr)
      );
    }
  });

  return navigationResult;
}

// 辅助函数：检查 HTML 字符串中所有 style 属性是否包含 background 或 color
export const hasBackgroundOrColor = (htmlString) => {
  if (!htmlString || typeof htmlString !== "string") return false;
  // 使用正则表达式匹配所有 style 属性
  const styleRegex = /style="([^"]*)"/g;
  let match;
  while ((match = styleRegex.exec(htmlString)) !== null) {
    const style = match[1];
    if (style.includes("background") || style.includes("color")) {
      return true;
    }
  }
  return false;
}

// 新增：移除 style 中 color 声明的函数
export const removeColorInStyle = (htmlString: string) => {
  // 把 style="…color:XXX;…" 中的 color:XXX; 删除
  return htmlString.replace(/(style="[^"]*?)\s*color:[^;"]*;?/g, "$1");
}

// 新增：递归清洗节点下所有 RichText
export const cleanRichText = (node: any) => {
  if (!node) return;
  if (node.type === "RichText" && typeof node.value === "string") {
    node.value = removeColorInStyle(node.value);
  }
  if (node.components) {
    Object.values(node.components).forEach(cleanRichText);
  }
  if (Array.isArray(node.items)) {
    node.items.forEach(cleanRichText);
  }
  if (Array.isArray(node.list)) {
    node.list.forEach(cleanRichText);
  }
}

// 新增：递归检查节点下所有 RichText 是否有 inline color
export const hasInlineColor = (node: any): boolean => {
  if (!node) return false;
  if (node.type === "RichText" && typeof node.value === "string") {
    return extractInlineColors(node.value).length > 0;
  }
  if (node.components) {
    return Object.values(node.components).some(hasInlineColor);
  }
  if (Array.isArray(node.items)) {
    return node.items.some(hasInlineColor);
  }
  if (Array.isArray(node.list)) {
    return node.list.some(hasInlineColor);
  }
  return false;
}

/**
   * 从一个 HTML 字符串中提取所有 inline color 值
   */
export const extractInlineColors = (html: string): string[] => {
  const reg = /color\s*:\s*([^;"]+)/gi;
  const result: string[] = [];
  let m: RegExpExecArray | null;
  while ((m = reg.exec(html))) {
    result.push(m[1].trim());
  }
  return result;
}

/**
 * 收集一个 RepeatableItem.components 下所有 RichText 的 inline color
 */
export const getItemRichTextColors = (item: any): string[] => {
  if (!item || !item.components) return [];

  const out: string[] = [];
  Object.values(item?.components ?? {}).forEach((comp: any) => {
    if (comp.type === "RichText" && typeof comp.value === "string") {
      out.push(...extractInlineColors(comp.value));
    }
  });
  return out;
}

/**
   * 如果 color 是 CSS 变量 var(--s-pre-color10)，提取数字部分 10
   */
export const extractCssVarIndex = (color: string): number | null => {
  const m = color?.match(/var\(--s-pre-color(\d+)\)/i);
  return m ? parseInt(m[1], 10) : null;
}

export const addColorToHtmlString = (html) => {
  /* use JSDOM start */
  const dom = new JSDOM(`<body>${html}</body>`);
  const { document, NodeFilter } = dom.window;

  const walker = document.createTreeWalker(
    document.body,
    NodeFilter.SHOW_TEXT
  );
  /* use JSDOM end */

  // const parser = new DOMParser();
  // const document = parser.parseFromString(html, 'text/html');
  // const walker = document.createTreeWalker(document.body, NodeFilter.SHOW_TEXT);

  const touched = new Set();
  // 匹配 color: #50555c（不区分大小写）
  const overrideColorRegex = /color\s*:\s*#50555c/gi;
  while (walker.nextNode()) {
    const textNode = walker.currentNode;
    if (!textNode?.nodeValue.trim()) continue;

    const parent = textNode.parentElement;
    if (!parent || touched.has(parent)) continue;
    touched.add(parent);

    const oriStyle = parent.getAttribute("style") || "";

    // 如果已有 color: #50555C，则替换成 color:rgb(10, 7, 7)
    if (overrideColorRegex.test(oriStyle)) {
      const newStyle = oriStyle.replace(overrideColorRegex, "color: #222222");
      parent.setAttribute("style", newStyle);
      continue;
    }

    // 如果已含 color 则跳过
    if (/color\s*:/i.test(oriStyle)) continue;

    // 根据 oriStyle 是否为空决定是否加前置分号
    let newStyle = oriStyle.trim();
    newStyle += "color: #222222;";
    parent.setAttribute("style", newStyle);
  }

  return document.body.innerHTML;
}

export const getBgValuePath = (patchPath: string): string | null => {
  const m = patchPath.match(
    /^(components\.repeatable1\.list\[\d+\]\.components)\.[^.]+$/
  );
  if (!m) return null;
  return `${m[1]}.background1.backgroundColor.value`;
}

export const getByPath = (obj: any, path: string): any => {
  return path.split(".").reduce((acc, key) => {
    const arr = key.match(/^(.+)\[(\d+)\]$/);
    if (arr) {
      const [, prop, idx] = arr;
      return acc?.[prop]?.[Number(idx)];
    }
    return acc?.[key];
  }, obj);
}