// spec: https://strikingly.atlassian.net/wiki/spaces/SPEC/pages/3893985371/L5+-+fix+the+custom+color+in+master+section+template
// @ts-nocheck
import { writeFileSync, readFileSync } from "fs";
import {
  hasBackgroundOrColor,
  removeColorInStyle,
  cleanRichText,
  hasInlineColor,
  extractInlineColors,
  getItemRichTextColors,
  extractCssVarIndex,
  addColorToHtmlString,
  getBgValuePath,
  getByPath,
  getNavigationInfo,
} from "./helper";


type CommandMode = 'detect' | 'extract' | 'fix' | 'extractNav';
const commandMode = (process.env.DETECTION_MODE as CommandMode) || 'extract';
const EXTRACT = commandMode === 'extract';
const DETECT_ONLY = commandMode === 'detect';

// 全量检测逻辑分离
function handleFullDiffNode(node: any, path: string[], output: any) {
  // 检查当前节点是否为 RichText
  if (node.type === "RichText") {
    const getInlineColors = extractInlineColors(node.value).join(',');
    output.result.push({
      RichText_type: "RichText",
      RichText_path: path.join('.'),
      RichText_color: getInlineColors,
      id: node.id || null
    });
  }

  // 检查当前节点是否为 Buttons，并将其内部每个 Button 当成独立 Button 处理
  if (node.type === "Buttons") {
    // 直接获取 block1 下的 items 数组
    const items = node.components.block1?.items || [];
    items.filter((btn: any) => btn.type === "Button").forEach((btn: any) => {
      output.result.push({
        Button_type: "Button",
        Button_path: path.join('.'),
        Button_color: btn.color,
        Button_backgroundSettings: btn.backgroundSettings || {},
        id: btn.id || null
      });
    });
  }

  // 检查当前节点是否为 Button
  if (node.type === "Button") {
    output.result.push({
      Button_type: "Button",
      Button_path: path.join('.'),
      Button_color: node.color,
      Button_backgroundSettings: node.backgroundSettings || {},
      id: node.id || null
    });
  }

  if (node.type === "SlideSettings" && node.layout_config?.hasOwnProperty("card")) {
    output.result.push({
      featureList_type: 'SlideSettings.layout_config.card',
      featureList_path: path.join('.'),
      featureList_card: node.layout_config.card,
      featureList_card_color: node.layout_config.card_color,
      featureList_border: node.layout_config.border,
      featureList_border_color: node.layout_config.border_color,
      featureList_cardBackgroundSettings: node.layout_config.cardBackgroundSettings || {},
      featureList_borderBackgroundSettings: node.layout_config.borderBackgroundSettings || {},
      id: node.id || null,
    })
  }

  // 检查 section 是否设置了 bg color
  if (node.type === "Background") {
    const useImage = node.useImage
    const useVideo = node.videoHtml
    if (useImage) {
      output.result.push({
        Background_textColor_type: "Background.textColor",
        Background_textColor: node.textColor || '',
        Background_textColor_path: path.join("."),
        id: node.id || null,
        Background_textColor_note: "The background is image",
      })
    } else if (useVideo) {
      output.result.push({
        Background_video_type: "Background.videoHtml",
        Background_video_path: path.join("."),
        id: node.id || null,
        Background_video_note: "The background is video",
      })
    } else {
      output.result.push({
        Background_color_type: "BackgroundColor",
        Background_color: node.backgroundColor?.value || '',
        Background_color_path: path.join("."),
        id: node.id || null,
        Background_color_note: "The background is background color",
      })
    }
  }
}

// 部分检测与修复逻辑分离
function handlePartialDiffAndFix(node: any, path: string[], output: any, { sectionData, themePreColors } = { sectionData: any, themePreColors: any }) {
  // new_grid 逻辑
  if (node.template_name === 'new_grid') {
    const list = node.components?.repeatable1?.list;
    if (Array.isArray(list)) {
      list.forEach((item) => {
        if (!item) {
          return;
        }

        const colors = getItemRichTextColors(item); // e.g. ['#222222','var(--s-pre-color10)']

        // 分离出纯 hex 的值和变量索引
        const hexColors = colors.filter((c) => /^#/.test(c.toLowerCase()));
        const varIndices = colors
          .map((c) => extractCssVarIndex(c))
          .filter((i): i is number => i !== null);
        const varColors = varIndices.map((idx) => {
          const themeColor = themePreColors.find((tc) => tc.key === idx);
          return themeColor?.value || "";
        });

        // 合并 hex 和变量颜色
        hexColors.push(...varColors);

        // 去重
        const uniqueHexColors = Array.from(new Set(hexColors));

        const hasDarkHex = uniqueHexColors.some((h) =>
          ["#000", "#000000"].includes(h.toLowerCase())
        );
        const hasWhiteHex = uniqueHexColors.some((h) =>
          ["#fff", "#ffffff"].includes(h.toLowerCase())
        );

        const hasImage = item?.components?.background1?.useImage;
        const bg = item?.components?.background1;

        // case ⑦ fix black 'custom color' text on image grid
        if (
          hasImage &&
          (bg?.textColor === "" ||
            bg?.textColor === "light" ||
            bg?.textColor === "dark") &&
          hasDarkHex
        ) {
          if (DETECT_ONLY) {
            output.result.push({
              type: "RepeatableItem.components.background1.textColor",
              path: path.join("."),
              id: item.id || null,
              note: "The text color is set to black when has image and the image overlay is not set to 'Dark text'.",
              whichCase: "case ⑦ fix black 'custom color' text on image grid",
            });
          } else {
            bg.textColor = "dark";
            // remove inline color from any RichText under this item
            cleanRichText(item);
          }
        }

        // case ⑧ fix closed card border
        if (
          hasImage &&
          (bg?.textColor === "light" || bg?.textColor === "overlay") &&
          hasWhiteHex
        ) {
          if (DETECT_ONLY) {
            output.result.push({
              type: "RepeatableItem.components.background1.textColor",
              path: path.join("."),
              id: item.id || null,
              note: "The text color is set to white when has image and the image overlay set to 'Light text' OR 'Overlay + Light Text'",
              whichCase: "case ⑧ fix closed card border",
            });
          } else {
            // remove inline color from any RichText under this item
            cleanRichText(item);
          }
        }

        // case ⑩ fix ‘use default’ text on fixed grid cell bg color
        if (
          !hasImage &&
          bg.backgroundColor?.value?.toLowerCase() === "#e8eaec"
        ) {
          Object.values(item?.components ?? {}).forEach((comp: any) => {
            if (comp.type === "RichText" && typeof comp.value === "string") {
              comp.value = addColorToHtmlString(comp.value);
            }
          });
        }
      });
    }
  }

  // feature list 逻辑
  if (
    node.type === "SlideSettings" &&
    node.layout_config?.hasOwnProperty("card")
  ) {
    // case ① fix closed card bg
    if (
      node.layout_config.card === false &&
      node.layout_config.cardBackgroundSettings?.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.cardBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The card bg is not set to use default when card is closed.",
          whichCase: "case ① fix closed card bg",
        });
      } else {
        node.layout_config.cardBackgroundSettings =
          node.layout_config.cardBackgroundSettings || {};
        node.layout_config.cardBackgroundSettings.type = "default";
      }
    }

    // case ② fix closed card border
    if (
      node.layout_config.border === false &&
      node.layout_config.borderBackgroundSettings?.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.borderBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The card border is not set to use default when border is closed.",
          whichCase: "case ② fix closed card border",
        });
      } else {
        node.layout_config.borderBackgroundSettings =
          node.layout_config.borderBackgroundSettings || {};
        node.layout_config.borderBackgroundSettings.type = "default";
      }
    }

    // case ③ keep the ‘custom color’ text on fixed grid cell bg color
    // Don't do anything, just keep the default

    // case ④ fix opened card bg white
    if (
      node.layout_config.card === true &&
      node.layout_config.card_color === "#ffffff" &&
      node.layout_config.cardBackgroundSettings?.type !== "custom"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.cardBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The card bg is not set to custom when card is opened and the card_color is set #ffffff .",
          whichCase: "case ④ fix opened card bg white",
        });
      } else {
        node.layout_config.cardBackgroundSettings =
          node.layout_config.cardBackgroundSettings || {};
        node.layout_config.cardBackgroundSettings.type = "custom";
      }
    }

    // case ⑤ fix opened card border gray
    if (
      node.layout_config.border === true &&
      node.layout_config.card_color === "#ffffff" &&
      node.layout_config.border_color === "#cccccc" &&
      node.layout_config.borderBackgroundSettings?.type !== "custom"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.borderBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The border color is not set to custom when border is opened and the card_color is set #ffffff and border_color is set #cccccc.",
          whichCase: "case ⑤ fix opened card border gray",
        });
      } else {
        node.layout_config.borderBackgroundSettings =
          node.layout_config.borderBackgroundSettings || {};
        node.layout_config.borderBackgroundSettings.type = "custom";
      }
    }

    // case ⑥ fix ‘custom color’ text on ‘use default' card bg
    if (
      node.layout_config.card === true &&
      (node.layout_config.cardBackgroundSettings?.type === "default" ||
        node.layout_config.cardBackgroundSettings?.default?.toLocaleLowerCase() ===
        node.layout_config.card_color?.toLocaleLowerCase())
    ) {
      const rpt = sectionData?.components?.repeatable1;
      if (rpt?.list) {
        rpt.list.forEach((item) => {
          if (DETECT_ONLY) {
            // only detect if there is any inline color in this item’s RichText
            const colors = getItemRichTextColors(item);
            if (colors.length > 0) {
              output.result.push({
                type: "RepeatableItem.components.RichText",
                path: path.join("."),
                id: item.id || null,
                note: "Remove custom-color text on default card background",
                whichCase:
                  "case ⑥ fix ‘custom color’ text on ‘use default' card bg color",
              });
            }
          } else {
            cleanRichText(item);
          }
        });
      }
    }

    // case ⑨ fix the wrong data of 'use default' card color
    if (
      node.layout_config.card === true &&
      !node.layout_config.card_color &&
      node.layout_config.cardBackgroundSettings.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.cardBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The card color is not set to use default when card is opened and the card_color is not set.",
          whichCase: "case ⑨ fix the wrong data of 'use default' card color",
        });
      } else {
        node.layout_config.cardBackgroundSettings =
          node.layout_config.cardBackgroundSettings || {};
        node.layout_config.cardBackgroundSettings.type = "default";
      }
    }

    // case ⑨ fix the wrong data of 'use default' card color
    if (
      node.layout_config.card === true &&
      node.layout_config.card_color?.toLowerCase() ===
      node.layout_config.cardBackgroundSettings?.default?.toLowerCase() &&
      node.layout_config.cardBackgroundSettings.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.cardBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The card color is not set to use default when card is opened and the card_color is set to the default value.",
          whichCase: "case ⑨ fix the wrong data of 'use default' card color",
        });
      } else {
        node.layout_config.cardBackgroundSettings =
          node.layout_config.cardBackgroundSettings || {};
        node.layout_config.cardBackgroundSettings.type = "default";
      }
    }

    // case ⑨ fix the wrong data of 'use default' card color
    if (
      node.layout_config.border === true &&
      !node.layout_config.border_color &&
      node.layout_config.borderBackgroundSettings.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.borderBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The border color is not set to use default when border is opened and the border_color is not set.",
          whichCase: "case ⑨ fix the wrong data of 'use default' card color",
        });
      } else {
        node.layout_config.borderBackgroundSettings =
          node.layout_config.borderBackgroundSettings || {};
        node.layout_config.borderBackgroundSettings.type = "default";
      }
    }

    // case ⑨ fix the wrong data of 'use default' card color
    if (
      node.layout_config.border === true &&
      node.layout_config.border_color?.toLowerCase() ===
      node.layout_config.borderBackgroundSettings?.default?.toLowerCase() &&
      node.layout_config.borderBackgroundSettings.type !== "default"
    ) {
      if (DETECT_ONLY) {
        output.result.push({
          type: "SlideSettings.layout_config.borderBackgroundSettings.type",
          path: path.join("."),
          id: node.id || null,
          note: "The border color is not set to use default when border is opened and the border_color is set to the default value.",
          whichCase: "case ⑨ fix the wrong data of 'use default' card color",
        });
      } else {
        node.layout_config.borderBackgroundSettings =
          node.layout_config.borderBackgroundSettings || {};
        node.layout_config.borderBackgroundSettings.type = "default";
      }
    }
  }

  // 检查当前节点是否为 RichText
  if (node.type === "RichText" && typeof node.value === "string" && hasBackgroundOrColor(node.value)) {
    let skip = false;

    if (sectionData.template_name === "new_grid") {
      const patchPath = path.join(".");
      const bgPath = getBgValuePath(patchPath);
      if (bgPath) {
        const bgValue = getByPath(sectionData, bgPath)?.toLowerCase() || "";
        const ignoreList = ["#ffffff", "#e8eaec", "#1c1c1c"];
        skip = ignoreList.includes(bgValue);
      }
    }

    if (!skip) {
      output.result.push({
        type: "RichText",
        path: path.join("."),
        id: node.id || null,
        note: "This RichText contains inline styles with background or color.",
      });
    }
  }

  // 检查当前节点是否为 Buttons，并将其内部每个 Button 当成独立 Button 处理
  if (node.type === "Buttons") {
    // 直接获取 block1 下的 items 数组
    const items = node.components.block1?.items || [];
    items.filter((btn: any) => btn.type === "Button").forEach((btn: any) => {
      if ((btn.color || (!btn.color && btn.hasOwnProperty("backgroundSettings"))) && !btn.backgroundSettings?.type) {
        output.result.push({
          parentType: "Buttons",
          type: "Button",
          path: path.join("."),
          id: btn.id || null,
          note: "The Button doesn't selected any option, like 'Use Default'、'Custom Color' or 'preset color'.",
        });
      }
    });
  }

  // 检查当前节点是否为 Button
  if (node.type === "Button") {
    if ((node.color || (!node.color && node.hasOwnProperty("backgroundSettings"))) && !node.backgroundSettings?.type) {
      output.result.push({
        type: "Button",
        path: path.join("."),
        id: node.id || null,
        note: "The Button doesn't selected any option, like 'Use Default'、'Custom Color' or 'preset color'.",
      });
    }
  }

  // 检查 section 是否设置了 bg color
  if (node.type === "Background") {
    const isUnderRepeatable1 = path.some(seg => seg.includes("components.repeatable1"))
    const useImage = node.useImage
    const useVideo = node.videoHtml
    if (!isUnderRepeatable1) {
      if (!useImage && !useVideo && node.backgroundColor?.value) {
        output.result.push({
          type: "BackgroundColor",
          path: path.join("."),
          id: node.id || null,
          note: `The background color is set to ${node.backgroundColor.value}.`,
        })
      }
    }
    if (useImage && !node.textColor) {
      output.result.push({
        type: "Background.textColor",
        path: path.join("."),
        id: node.id || null,
        note: "The background has image but textColor is not set.",
      })
    }
  }
}

function parseJsonForStyles(
  themePreColors,
  sectionData,
  sectionIdx
) {
  const output = { sectionIndex: sectionIdx, result: [] };

  function traverse(node: any, path: string[] = []) {
    if (!node) return;

    if (EXTRACT) {
      handleFullDiffNode(node, path, output);
    } else {
      handlePartialDiffAndFix(node, path, output, { themePreColors, sectionData });
    }

    // 遍历子节点
    if (node.components && node.type !== "Buttons") {
      // Only traverse individual Button when no Buttons group is present
      let componentKeys = Object.keys(node.components);
      if (componentKeys.includes('buttons') && componentKeys.includes('button1')) {
        componentKeys = componentKeys.filter(k => k !== 'button1');
      }
      componentKeys.forEach((key) => {
        const child = node.components[key];
        traverse(child, [...path, `components.${key}`]);
      });
    }

    // 遍历 items 数组（如果存在）
    if (Array.isArray(node.items)) {
      node.items.forEach((item, index) => {
        traverse(item, [...path, `items[${index}]`]);
      });
    }

    // 遍历 list 数组（如果存在，例如 repeatable1.list）
    if (Array.isArray(node.list)) {
      node.list.forEach((item, index) => {
        traverse(item, [...path, `list[${index}]`]);
      });
    }
  }

  // 开始遍历 JSON 数据
  traverse(sectionData);

  return output;
}

export function mainParse(jsonData) {
  const themePreColors = jsonData.customColors?.themePreColors || [];
  const result = [];
  // 生成导航信息
  const navigationInfo = getNavigationInfo(jsonData.navigation.items);

  // 遍历 pages 数组
  const navLinks = jsonData.navigation.items.reduce((pre, cur) => {
    if (cur.items) {
      return pre.concat(cur.items);
    }
    return pre;
  }, []);

  const pages = navLinks.map((item) => {
    return jsonData.pages.find((page) => page.uid === item.id);
  });

  if (Array.isArray(pages)) {
    pages.forEach((page, pageIndex) => {
      // 遍历 sections 数组
      if (Array.isArray(page.sections)) {
        page.sections.forEach((section, sectionIndex) => {
          const sectionResult = parseJsonForStyles(
            themePreColors,
            section,
            sectionIndex
          );
          if (sectionResult.result?.length) {
            result.push({
              pageIndex,
              pageUid: page.uid,
              pageTitle: page.title,
              sectionIndex: sectionIndex,
              result: sectionResult.result,
            });
          }
        });
      }
    });
  }

  return { jsonData, result, navigationInfo };
}