import { sendSlackMessage } from '../clients/slack';
import { readdirSync, readFileSync } from 'fs';
import { join } from 'path';

async function main() {
  // Scan both diff directories for JSON files, with error handling
  const dirs = [
    join(__dirname, 'diffWithBaseline'),
    join(__dirname, 'diffAIWithBaseline')
  ];
  let filePaths: string[] = [];
  dirs.forEach(dir => {
    try {
      const jsFiles = readdirSync(dir).filter(f => f.endsWith('.json'));
      jsFiles.forEach(f => filePaths.push(join(dir, f)));
    } catch (e) {
      console.warn(`Directory ${dir} not found, skipping.`);
    }
  });
  // Determine if any diff file contains non-empty array data
  const hasChanges = filePaths.some(fp => {
    try {
      const data = JSON.parse(readFileSync(fp, 'utf-8'));
      return Array.isArray(data) && data.length > 0;
    } catch {
      return false;
    }
  });
  if (!hasChanges) {
    console.log('No changes detected in both diff directories, skipping Slack notification.');
    process.exit(0);
  }

  const webhookUrl = process.env.SLACK_WEBHOOK!;
  const nonAiUrl = `https://docs.google.com/spreadsheets/d/${process.env.GOOGLE_SHEET_ID}/edit`;
  const aiUrl = `https://docs.google.com/docs.google.com/spreadsheets/d/${process.env.GOOGLE_SHEET_AI_ID}/edit`;
  await sendSlackMessage(webhookUrl, {
    issue: 'Inconsistencies are detected in i18n master section template',
    details: `Google Sheet Links:\n- Non-AI selectors: ${nonAiUrl}\n- AI selectors: ${aiUrl}`
  });
}

main().catch(err => {
  console.error(err);
  process.exit(1);
});
