// @ts-nocheck
import { google } from 'googleapis';
import { readdirSync, readFileSync } from 'fs';
import { join } from 'path';

async function uploadAllToSheets() {
  const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';
  const spreadsheetId = useAiSelectors
    ? process.env.GOOGLE_SHEET_AI_ID!
    : process.env.GOOGLE_SHEET_ID!;
  const keyFile = process.env.GOOGLE_SERVICE_ACCOUNT_KEY_FILE!;
  const auth = new google.auth.GoogleAuth({
    keyFile,
    scopes: ['https://www.googleapis.com/auth/spreadsheets'],
  });
  const sheets = google.sheets({ version: 'v4', auth });
  
  const dir = join(__dirname, useAiSelectors ? 'diffAIWithBaseline' : 'diffWithBaseline');
  const files = readdirSync(dir).filter(f => f.endsWith('.json'));

  // Fetch existing sheet names
  const { data: spreadsheet } = await sheets.spreadsheets.get({ spreadsheetId });
  const existing = spreadsheet.sheets?.map((s: any) => s.properties?.title || '') || [];

  for (const file of files) {
    const sheetName = file.replace(/\.json$/, '');
    if (!existing.includes(sheetName)) {
      // Create new sheet
      await sheets.spreadsheets.batchUpdate({
        spreadsheetId,
        requestBody: {
          requests: [{ addSheet: { properties: { title: sheetName } } }]
        }
      });
    }

    const raw = readFileSync(join(dir, file), 'utf-8');
    const data = JSON.parse(raw);
    if (!data.length) continue;
    const headers = Object.keys(data[0]);
    const rows = data.map((row: any) =>
      headers.map((h: string) => {
        const cell = row[h];
        return cell != null && typeof cell === 'object'
          ? JSON.stringify(cell)
          : cell;
      })
    );

    // Clear existing values
    await sheets.spreadsheets.values.clear({ spreadsheetId, range: `'${sheetName}'` });
    // Write headers and data
    await sheets.spreadsheets.values.append({
      spreadsheetId,
      range: `'${sheetName}'!A1`,
      valueInputOption: 'RAW',
      requestBody: { values: [headers, ...rows] }
    });

    console.log(`✅ Uploaded ${file} to tab ${sheetName}`);
  }
}

async function main() {
  try {
    await uploadAllToSheets();
  } catch (err) {
    console.error('Error in task:gsheet workflow:', err);
    process.exit(1);
  }
}

main();
