import { writeFileSync, mkdirSync, existsSync, rmSync } from 'fs'
import { fetchSiteData } from '../clients/bobcat/SiteInfo'
import {
    section_selectors,
    ai_section_selectors,
} from '../constant/section_selectors'
import { mainParse } from './handlePageContent'

/* 
    detect: 设置限制条件，获取指定有问题的字段
    extract: 提出整个 page content 指定的字段数据
    fix: 修复 page content 数据，修复后获取完整的 page content
    extractNav: 获取导航信息，可以查看下拉菜单有多少个
*/

type CommandMode = 'detect' | 'extract' | 'fix' | 'extractNav';

const commandMode = (process.env.DETECTION_MODE as CommandMode) || 'extract';
const useAiSelectors = process.env.USE_AI_SELECTORS === 'true';

// 根据环境变量选择使用哪个 selector 数组
const selectedSelectors = useAiSelectors ? ai_section_selectors : section_selectors;

// 确保输出目录存在
let outDir = ''
if (commandMode === 'detect') {
    outDir = 'src/detect_section_selector_masters/detectOutput'
} else if (commandMode === 'extract') {
    outDir = 'src/detect_section_selector_masters/extractOutput'
} else if (commandMode === 'fix') {
    outDir = 'src/detect_section_selector_masters/fixedOutput'
} else if (commandMode === 'extractNav') {
    outDir = 'src/detect_section_selector_masters/extractNavOutput'
}

rmSync(outDir, { recursive: true, force: true })

// Ensure output directory exists
mkdirSync(outDir, { recursive: true })

await Promise.all(
    selectedSelectors.map(async (siteId, index) => {
        const { content } = await fetchSiteData(siteId)
        const { jsonData: pageContent, result, navigationInfo } = mainParse(content) || {}
        const file = `${outDir}/${siteId}_${commandMode}.json`

        let data = []
        if (commandMode === 'detect' || commandMode === 'extract') {
            data = result
        } else if (commandMode === 'fix') {
            data = pageContent
        } else if (commandMode === 'extractNav') {
            data = navigationInfo
        }
        writeFileSync(file, JSON.stringify(data, null, 2), 'utf-8')
        console.log(`✅ ${siteId} output written to ${file} ${useAiSelectors ? '(AI selectors)' : ''}`)
    })
)