import { spawn } from "bun";
import Redis from "ioredis";

// 任务执行入口，自动设置 SLACK_WEBHOOK 并依次运行 gsheet、gsheet:ai 和 notify-slack
async function processDetect() {
  const env = {
    ...process.env
  };

  // 运行 bun run detect
  await spawn({
    cmd: ["bun", "run", "detect"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });

  // 运行 bun run task:gsheet
  await spawn({
    cmd: ["bun", "run", "task:gsheet"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });

  // 延迟执行 task:gsheet:ai 避免 Google Sheet 限流
  const delayMs = parseInt(process.env.GSHEET_AI_DELAY_MS ?? "60000", 10);
  console.log(`Waiting ${delayMs}ms before running task:gsheet:ai to avoid rate limiting`);
  await new Promise<void>(resolve => setTimeout(resolve, delayMs));

  // 运行 bun run task:gsheet:ai
  await spawn({
    cmd: ["bun", "run", "task:gsheet:ai"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });

  // 运行 bun run notify-slack
  await spawn({
    cmd: ["bun", "run", "notify-slack"],
    env,
    stdout: "inherit",
    stderr: "inherit"
  });
}

function isMonday11AM() {
  const now = new Date();
  return now.getDay() === 1 && now.getHours() === 11;
}

async function main() {
  const redis = new Redis(Bun.env.REDIS_URL);
  const shouldRun = await redis.get("falcon:should_run_tasks");

  if (shouldRun === "true") {
    await processDetect();
    await redis.del("falcon:should_run_tasks");
    console.log("should_run_tasks 执行完毕，已清理 redis key。");
  } else if (isMonday11AM()) {
    const mondayKey = "falcon:monday_11am_executed";
    const alreadyExecuted = await redis.get(mondayKey);
    if (alreadyExecuted) {
      console.log("本周一11点任务已执行，跳过。");
    } else {
      await processDetect();
      // 设置标记，24小时后自动过期
      await redis.set(mondayKey, "true", "EX", 60 * 60 * 24);
      console.log("定时任务触发，已执行 main 逻辑，并设置本周标记。");
    }
  } else {
    console.log("should_run_tasks 不是 'true'，且不是定时任务，跳过。");
  }

  await redis.quit();
}

// 调用入口
main().catch(err => {
  console.error(err);
  process.exit(1);
});