import React from 'react'
import { connect } from 'react-redux'
import PropTypes from 'prop-types'
import DOM from 'react-dom-factories'
import Immutable from 'immutable'
import Morearty from 'morearty'
import reactMixin from 'react-mixin'
import TimerMixin from 'react-timer-mixin'
import ReactTransitionGroup from 'react-transition-group/TransitionGroup'
import classNames from 'classnames'
import { tct } from 'r-i18n'
import { Icon } from 'component-kit'

import PremiumFeature from './PremiumFeature'
import EditorActions from 'js/actions/editor_actions'
import SiteConnectionActions from 'js/actions/SiteConnectionActions'
import EditorConstants from 'js/constants/editor_constants'
import EditorDispatcher from 'js/dispatcher/editor_dispatcher'

import ConfStore from 'js/stores/conf_store'
import CurrentUserStore from 'js/stores/current_user_store'
import CollaboratorsStore from 'js/stores/collaborators_store'
import PageMetaStore from 'js/stores/page_meta_store'
import UserMetaStore from 'js/stores/user_meta_store'

import TooltipMixin from 'js/utils/mixins/tooltip_mixin'

import PurchaseContainer from 'js/components/domain_purchase/PurchaseContainer'

import JQSlide from 'js/components/helpers/jqslide'

import StrikinglyOrSxl from 'js/components/StrikinglyOrSxl'

import _ from 'lodash'

import { openDialog } from 'nextgen/shared/actions/dialogsActions'

const parseDomainPromise = import('parse-domain')

// 1 has a recored
// 2 has forwarind
// 3 both or others
const _checkConfig = {
  '1&1': 2,
  '101 Domain': 1,
  '123-reg': 2,
  Bluehost: 2,
  cPanel: 2,
  CrazyDomains: 1,
  Gandi: 2,
  GoDaddy: 2,
  'Microsoft 365': 1,
  Namecheap: 2,
  'Network Solutions': 1,
  OVH: 1,
  'Register.com': 2,
  Wix: 1,
  'Wordpress.com': 1,
  'Yahoo!': 1,
  Others: 3,
  新网: 1,
  阿里云万网: 1,
  'Nic.cl': 1,
  Amen: 1,
  'Sakura Internet（さくらインターネット）': 1,
  'Muu Muu (ムームードメイン)': 1,
  'Value-Domain': 2,
  XSERVER: 2,
}

@reactMixin.decorate(Morearty.Mixin)
@reactMixin.decorate(TooltipMixin.enableAfterUpdate())
@reactMixin.decorate(TimerMixin)
class DomainsTab extends React.Component {
  static propTypes = {
    permalink: PropTypes.string.isRequired,
    customDomain: PropTypes.string,
    isDirLink: PropTypes.bool,
    binding: PropTypes.object.isRequired,
    onSaveSettings: PropTypes.func.isRequired,
    selected: PropTypes.bool,
    currentTabName: PropTypes.string,
    changeTabName: PropTypes.func,
    showQrCode: PropTypes.func,
  }

  constructor(props) {
    super(props)
    this._onConnectDomainAfterRegister = this._onConnectDomainAfterRegister.bind(
      this,
    )
    this._onRegisterNewDomain = this._onRegisterNewDomain.bind(this)
    this._onRegisterNewDomainWithName = this._onRegisterNewDomainWithName.bind(
      this,
    )
    this._onChangeStep = this._onChangeStep.bind(this)
    this._onBackToTab = this._onBackToTab.bind(this)
    this._removePurchaseSuccessMsg = this._removePurchaseSuccessMsg.bind(this)
    this._onChangeCustomDomain = this._onChangeCustomDomain.bind(this)
  }

  componentWillMount() {
    this.setStateWithForceUpdate(this.props)
  }

  componentDidMount() {
    if (['editor', 'blogger'].includes(CollaboratorsStore.getRole())) {
      return
    }

    // Hack: category and name can be updated from publish dialog, update states when open setting dialog
    this._openPageSettingsToken = EditorDispatcher.register(payload => {
      switch (payload.actionType) {
        case EditorConstants.ActionTypes.OPEN_PAGE_SETTINGS:
          this.setStateWithForceUpdate(this.props)
          // refresh status when domain tab is selected after opened settings dialog
          if (this.props.selected) {
            this._refreshDomainStatus()
          }
      }
    })
  }

  componentWillUnmount() {
    EditorDispatcher.unregister(this._openPageSettingsToken)
  }

  componentDidUpdate(prevProps, prevState) {
    // refresh status when swtich to domain tab
    if (!prevProps.selected && this.props.selected) {
      this._refreshDomainStatus()
    }
  }

  getDefaultState() {
    return Immutable.Map({
      // global common tab state
      isUpdating: false,
      isSaved: false,
      status: '',
      // domains state
      showRegisterBtn: true,
      showPurchaseSuccessMsg: false,
      customDomainSaved: false,
      customDomainStatus: '',
      customDomainMessage: '',
      customDomainCheckRs: '',
      domainProvider:
        window.localStorage.getItem('_strk_domain_provider') || '',
      currentStep: 'tab',
      currentDomain: this.props.customDomain || '',
      domainPurchaseStatus: CurrentUserStore.domainPurchaseProcessStatus(),
      v2DomainSetting: (() => {
        const ds = PageMetaStore.getUserV2DomainSetting()
        return ds ? ds.toJS() : null
      })(),
    })
  }

  // because morearty's shouldComponentUpdate doesn't account for changes in state
  setStateWithForceUpdate(h) {
    this.setState(h)
    this.forceUpdate()
  }

  _refreshDomainStatus() {
    const customDomainStatus = this.getDefaultBinding().get(
      'customDomainStatus',
    )
    let checkRs = this.getDefaultBinding().get('customDomainCheckRs')

    if (!customDomainStatus && checkRs) {
      checkRs = checkRs.toJS()
      if (checkRs.correct === false) {
        this._onCheckDomainStatus()
      }
    }
  }

  _onPermalinkChange(e) {
    this.setStateWithForceUpdate({
      permalink: e.target.value,
    })
  }

  _onBackToTab() {
    this._onChangeStep('tab')
    this.props.changeTabName(__('EditorSettings|Domain'))
  }

  _onConnectDomainAfterRegister(newDomain) {
    // free user won't connect domain
    if (CurrentUserStore.isFreeMembership()) {
      this.getDefaultBinding().set('showPurchaseSuccessMsg', true)
      this._onBackToTab()
      return
    }

    //
    EditorActions.addDomainToPool(newDomain)
    newDomain = `www.${newDomain}`

    EditorActions.updateCustomDomain(newDomain)
    this.getDefaultBinding().set('currentDomain', newDomain)

    // TODO change that to connect / disconnect later
    SiteConnectionActions.updateCustomDomain({
      siteId: this.props.siteId,
      domain: newDomain,
      success: data => {
        this.props.updateV2CustomDomain(data)
        this.getDefaultBinding().set('customDomainStatus', 'success')
      },
    })

    this.getDefaultBinding().set('showPurchaseSuccessMsg', true)
    this.getDefaultBinding().set('domainPurchaseStatus', 'buy')
    this.getDefaultBinding().set('justConnectedAfterRegister', true)
    this.getDefaultBinding().set('showRegisterBtn', false)
    this.setTimeout(() => {
      this._onBackToTab()
    }, 1500)
  }

  _removePurchaseSuccessMsg() {
    this.getDefaultBinding().set('showPurchaseSuccessMsg', false)
  }

  _onSavePermalink = () => {
    if (this.refs.permalink.value.length >= 63) {
      this.getDefaultBinding().set(
        'status',
        __('EditorSettings|Domain name cannot be longer than 63 characters.'),
      )
    } else {
      this.getDefaultBinding().set('status', '')
      this.props.onSaveSettings({
        permalink: this.refs.permalink.value,
      })
    }
  }

  _saveDomainProvider(provider) {
    this.getDefaultBinding().set('domainProvider', provider)
    window.localStorage.setItem('_strk_domain_provider', provider)
  }

  _clearDomainProvider() {
    this.getDefaultBinding().set('domainProvider', 'default')
    window.localStorage.removeItem('_strk_domain_provider')
  }

  _onRegisterNewDomain() {
    this._onChangeStep('purchase')
    this._initialDomainName = ''
    // EditorActions.registerDomain()
  }

  _onRegisterNewDomainWithName() {
    this._onChangeStep('purchase')
    this._initialDomainName = this.getDefaultBinding().get('currentDomain')
  }

  _onChangeStep(step) {
    this.getDefaultBinding().set('currentStep', step)
  }

  _onChangeCustomDomain(e) {
    this.getDefaultBinding().set('currentDomain', e.target.value)
  }

  _onSaveCustomDomain = e => {
    let currentDomain = this.getDefaultBinding()
      .get('currentDomain')
      .toLowerCase()
    if (
      ConfStore.getIsSxl() &&
      !currentDomain &&
      !window.confirm(
        '若停止使用自定义域名，网站将临时下线。你需要发布审核，通过后即可访问。确定停止使用自定义域名？',
      )
    ) {
      return
    }
    this.getDefaultBinding().set('saveBtnClicked', true)
    const customDomainStatus = this.getDefaultBinding().get(
      'customDomainStatus',
    )
    const { v2Domains } = this.props

    if (['updating', 'checking'].includes(customDomainStatus)) {
      return
    }

    this._clearDomainProvider()
    this.setState({ currentDomain })

    // If current domain is empty, we don't even bother, it's a disconnect
    // If current domain start with www., we don't need to test (unless it's connecting to www.com)
    if (!currentDomain || currentDomain.startsWith('www.')) {
      this._setUpdating(currentDomain)
      this._updateV2CustomDomain(currentDomain)
      return
    }

    const existingDomain = _.find(v2Domains, d => d.name == currentDomain)

    if (!existingDomain) {
      this.getDefaultBinding().merge(
        Immutable.fromJS({
          customDomainStatus: 'updating',
        }),
      )

      Promise.all([parseDomainPromise]).then(([parseDomain]) => {
        const domainChunks = parseDomain(currentDomain)

        if (domainChunks && !domainChunks.subdomain) {
          currentDomain = `www.${currentDomain}`
        }

        this.getDefaultBinding().merge(
          Immutable.fromJS({
            currentDomain,
          }),
        )
        this._updateV2CustomDomain(currentDomain)
      })

      return
    }

    if (existingDomain.is_wwwizered) {
      currentDomain = `www.${currentDomain}`
    }

    this._setUpdating(currentDomain)
    this._updateV2CustomDomain(currentDomain)
  }

  _setUpdating(currentDomain) {
    this.getDefaultBinding().merge(
      Immutable.fromJS({
        customDomainStatus: 'updating',
        currentDomain,
      }),
    )
  }

  _updateV2CustomDomain(fqdn) {
    // TODO change that to connect / disconnect later
    SiteConnectionActions.updateCustomDomain({
      siteId: this.props.siteId,
      domain: fqdn,
      success: data => {
        this.props.updateV2CustomDomain(data)
        this.getDefaultBinding().set('customDomainStatus', 'success')
      },
      afterError: res => {
        this.getDefaultBinding().set('customDomainStatus', '')
        this.getDefaultBinding().set('currentDomain', '')
      },
    })
  }

  _onCheckDomainStatus() {
    this.getDefaultBinding().set('customDomainStatus', 'checking')
    EditorActions.checkDomainStatus()
  }

  _onClickshowRegisterBtn() {
    this.getDefaultBinding().set('showRegisterBtn', true)
  }

  _onChangeProvider(e) {
    this._saveDomainProvider(e.target.value)
  }

  _renderDomainSettingsBox(v2Domain = false) {
    const status = this.getDefaultBinding().get('customDomainStatus')
    let checkRs = this.getDefaultBinding().get('customDomainCheckRs')
    let linkUrl = ''

    if (!v2Domain) {
      if (checkRs && checkRs.toJS) {
        checkRs = checkRs.toJS()
      }

      if (
        typeof checkRs !== 'object' ||
        checkRs.correct ||
        !checkRs.domainRegistered ||
        status === 'updating'
      ) {
        return {
          renderRs: null,
          showDomainSettingsStatus: false,
        }
      }
    }

    const domainProviders = ConfStore.getDomainSupportedProvider()
    const currentProvider = this.getDefaultBinding().get('domainProvider')
    const saveBtnClicked = this.getDefaultBinding().get('saveBtnClicked')
    const currentProviderObj = domainProviders.find(
      provider => provider.name === currentProvider,
    )
    const subdomainUrl = ConfStore.getIsSxl()
      ? 'http://help.sxl.cn/hc/zh-cn/articles/214721838'
      : 'http://support.strikingly.com/hc/en-us/articles/214364738-Generic-Subdomain-Setup-Tutorial'

    let checkType = 'arecord'

    if (currentProviderObj) {
      switch (_checkConfig[currentProviderObj.name]) {
        case 2:
          checkType = 'forwarding'
          break
        case 3:
          checkType = 'forwarding_or_arecord'
          break
        // no default
      }
    }

    function getCheckDetail(type, index) {
      let text = ''
      let success = true

      if (checkRs.domainType === 'non_www_subdomain') {
        if (
          !checkRs.nonWwwSubdomainHasCname ||
          !checkRs.nonWwwSubdomainHasCnameToRoot
        ) {
          success = false
        }
        text = __(
          'EditorSettings|Add a CNAME record pointing from <strong>%{subDomain}</strong> to <strong>dns.strikingly.com</strong>',
          { subDomain: checkRs.nonWwwSubdomain },
        )
      } else {
        switch (type) {
          case 'cname':
            if (
              !checkRs.wwwSubdomainHasCname ||
              !checkRs.wwwSubdomainHasCnameToRoot
            ) {
              success = false
            }
            text = __(
              'EditorSettings|Add a CNAME record pointing from <strong>www</strong> to <strong>%{rootDomain}</strong>',
              { rootDomain: 'dns.strikingly.com' },
            )
            break
          case 'arecord':
            if (
              !checkRs.rootDomainRedirected ||
              checkRs.rootDomainHasHerokuIp ||
              checkRs.wwwSubdomainHasHerokuIp
            ) {
              success = false
            }
            text = __(
              'EditorSettings|Add an A record pointing from <strong>@</strong> to <strong>54.183.102.22</strong>',
            )
            break
          case 'forwarding':
            if (!checkRs.rootDomainRedirected) {
              success = false
            }
            text = __(
              'EditorSettings|Forward %{rootDomain} to <strong>http://%{wwwSubdomain}</strong>',
              {
                rootDomain: checkRs.rootDomain,
                wwwSubdomain: checkRs.wwwSubdomain,
              },
            )
            break
          case 'forwarding_or_arecord':
            if (
              !checkRs.rootDomainRedirected ||
              checkRs.rootDomainHasHerokuIp ||
              checkRs.wwwSubdomainHasHerokuIp
            ) {
              success = false
            }
            text = __(
              'EditorSettings|Forward %{rootDomain} to <strong>http://%{wwwSubdomain}</strong> <br> <strong>OR</strong> Add an A record pointing from <strong>@</strong> to <strong>54.183.102.22</strong>',
              {
                rootDomain: checkRs.rootDomain,
                wwwSubdomain: checkRs.wwwSubdomain,
              },
            )
            break
          // no default
        }
      }

      return (
        <tr key={type}>
          <td className="domain-check-result-item">
            &#8226;&nbsp;
            <span dangerouslySetInnerHTML={{ __html: text }} />
          </td>
          <td className="domain-check-result-tag">
            {status === 'checking' ? (
              <i className="fa fa-spinner fa-pulse" />
            ) : (
              <span
                className={classNames('s-box-tag', {
                  red: !success,
                  green: success,
                })}>
                {success
                  ? __('EditorSettings|Complete')
                  : __('EditorSettings|Pending')}
              </span>
            )}
          </td>
        </tr>
      )
    }

    if (currentProviderObj) {
      linkUrl =
        checkRs.domainType === 'non_www_subdomain'
          ? currentProviderObj.subdomain || subdomainUrl
          : currentProviderObj.url
    }

    __('The page has an invalid domain.')

    // if (v2Domain) {
    //   currentProviderObj = domainProviders.find((provider) => provider.name === 'Others')
    //   linkUrl = currentProviderObj.url
    // }

    return {
      showDomainSettingsStatus: true,
      renderRs: (
        <ReactTransitionGroup>
          <JQSlide component={DOM.div} className="domain-check-box s-box last">
            <div>
              <div className="field-title">
                {saveBtnClicked
                  ? __(
                      'EditorSettings|Almost done! You still need to configure your domain settings.',
                    )
                  : __(
                      'EditorSettings|Note: Make sure your domain is configured properly.',
                    )}
              </div>
              <select
                onChange={this._onChangeProvider.bind(this)}
                value={currentProvider}>
                <option value="default" disabled={true}>
                  {__('EditorSettings|Where did you register this domain?')}
                </option>
                {domainProviders.map((item, index) => (
                  <option key={item.name} value={item.name}>
                    {item.name === 'Others' ? __('Domain|Others') : item.name}
                  </option>
                ))}
              </select>
              {currentProviderObj && (
                <div>
                  <ReactTransitionGroup>
                    <JQSlide
                      component={DOM.div}
                      className="domain-check-result">
                      <div style={{ marginBottom: '12px' }} />
                    </JQSlide>
                  </ReactTransitionGroup>
                  <div>
                    <a
                      className="s-btn basic-blue"
                      target="_blank"
                      href={linkUrl}>
                      <i className="fa fa-book left-icon" />
                      {__('EditorSettings|Read Tutorial')}
                    </a>
                    <i
                      className="fa fa-question-circle"
                      rel="tooltip-right"
                      data-original-title={__(
                        'EditorSettings|Changes in domain settings usually take effect in a few hours, but can take up to 2 days.',
                      )}
                    />
                  </div>
                </div>
              )}
            </div>
          </JQSlide>
        </ReactTransitionGroup>
      ),
    }
  }

  _renderDomainTab() {
    const {
      isUpdating,
      status,
      isSaved,
      customDomainSaved,
      customDomainStatus,
      customDomainMessage,
      customDomainCheckRs,
      showRegisterBtn,
      showPurchaseSuccessMsg,
      domainPurchaseStatus,
      justConnectedAfterRegister,
      currentDomain,
    } = this.getDefaultBinding()
      .get()
      .toJS()
    const { permalink, isDirLink } = this.state
    const { v2DomainSetting } = this.props
    const domainSupported = /(.com|.net|.org|.me|.co)$/.test(currentDomain)
    const isSxl = ConfStore.getIsSxl()
    const v2DomainConnection = PageMetaStore.getV2DomainConnection()
    const showConnectedMessage =
      v2DomainConnection && v2DomainConnection.get('domain_id')
    const hasNoDomainId =
      v2DomainConnection && !v2DomainConnection.get('domain_id')
    const isSiteOfResellerClient = PageMetaStore.isSiteOfResellerClient()
    const canBuyAllDomains = ConfStore.getCanBuyAllDomains()

    let registerDomainContainer
    if (!isSxl) {
      // rollout for new domain project v1
      if (!canBuyAllDomains) {
        if (domainPurchaseStatus === 'free') {
          registerDomainContainer = (
            <div className="register-domain">
              {tct(
                __(
                  "If you [link: purchase a yearly subscription plan] to Strikingly, we'll give you a domain for free for the first year!",
                ),
                {
                  root: <div className="field-hint" />,
                  link: (
                    <a
                      href={
                        UserMetaStore.isZbjUser()
                          ? 'javascript: alert("八戒云站暂不支持升级，请添加新的套餐。")'
                          : '/s/pricing?ref=free_domain'
                      }
                    />
                  ),
                },
              )}
              <div
                className="s-btn big basic-blue"
                onClick={this._onRegisterNewDomain}>
                {__('Check Availability')}
              </div>
            </div>
          )
        } else if (
          !justConnectedAfterRegister &&
          v2DomainSetting.entitledToFreeDomain
        ) {
          registerDomainContainer = (
            <div className="register-domain">
              <div className="field-hint">
                {__(
                  `You're on a yearly plan and can register a new domain for free!`,
                )}
              </div>
              <div
                className="s-btn big basic-blue"
                onClick={this._onRegisterNewDomain}>
                {__('Claim My Free Domain!')}
              </div>
            </div>
          )
        } else {
          registerDomainContainer = (
            <div className="register-domain">
              <div className="field-hint">
                {__(`Don't own a domain yet? Grab one here for $24.95/year.`)}
              </div>
              <div
                className="s-btn big basic-blue"
                onClick={this._onRegisterNewDomain}>
                {__('Register New Domain')}
              </div>
            </div>
          )
        }
      } else if (
        CurrentUserStore.hasFreeDomain() &&
        v2DomainSetting.entitledToFreeDomain
      ) {
        registerDomainContainer = (
          <div className="register-domain">
            <div className="field-hint">
              {__(
                `You're on a yearly plan and can register a new domain for free!`,
              )}
            </div>
            <div
              className="s-btn big basic-blue"
              onClick={this._onRegisterNewDomain}>
              {__('Claim My Free Domain!')}
            </div>
          </div>
        )
      } else {
        registerDomainContainer = (
          <div className="register-domain">
            <div className="field-hint">
              {__(`Don't own a domain yet? Grab one here.`)}
            </div>
            <div
              className="s-btn big basic-blue"
              onClick={this._onRegisterNewDomain}>
              {__('Register New Domain')}
            </div>
          </div>
        )
      }
    }

    let permalinkInner
    let completedPermalink
    if (isDirLink) {
      completedPermalink = `${
        isSxl ? 'http://www.sxl.cn/' : 'http://www.strikingly.com/'
      }${permalink}`
      permalinkInner = (
        <div className="permalink-inner">
          {isSxl ? 'http://www.sxl.cn/' : 'http://www.strikingly.com/'}
          <input
            type="text"
            ref="permalink"
            value={permalink}
            onChange={e => {
              // permalink could be updated from prepublish dialog, shouldn't use defaultValue here.
              this._onPermalinkChange(e)
            }}
          />
        </div>
      )
    } else {
      completedPermalink = `http://${permalink}${
        isSxl ? '.sxl.cn/' : '.strikingly.com/'
      }`
      permalinkInner = (
        <div className="permalink-inner">
          http://
          <input
            type="text"
            ref="permalink"
            value={permalink}
            onChange={e => {
              // permalink could be updated from prepublish dialog, shouldn't use defaultValue here.
              this._onPermalinkChange(e)
            }}
          />
          {isSxl ? '.sxl.cn/' : '.strikingly.com/'}
        </div>
      )
    }

    const domainSettingsRs = this._renderDomainSettingsBox(hasNoDomainId)
    const MoreartyInput = Morearty.DOM.input

    return (
      <div
        key="first-step"
        className="page-settings-content s-dialog-content domains-tab">
        {!isSxl && (
          <div>
            {showPurchaseSuccessMsg && (
              <ReactTransitionGroup>
                <JQSlide component={DOM.div} className="s-box green small fist">
                  <i className="fa fa-check" />
                  {tct(
                    __(
                      'Domain|[placeholderStrong: You just got a new domain!]',
                    ),
                    {
                      root: <span />,
                      placeholderStrong: <strong />,
                    },
                  )}
                  &nbsp;
                  {!CurrentUserStore.isFreeMembership() &&
                    __("Domain|And we've already connected it to your site.")}
                  {__(
                    'Domain|Remember, you must check your email to validate your domain.',
                  )}
                  &nbsp;
                  <a
                    href={v2DomainSetting ? '/s#/v2_domains/' : '/s#/domains'}
                    target="_blank">
                    {__('Domain|View your new domain in domain dashboard.')}
                  </a>
                  <div
                    className="close-btn"
                    onClick={this._removePurchaseSuccessMsg}>
                    ×
                  </div>
                </JQSlide>
              </ReactTransitionGroup>
            )}
            {!isSiteOfResellerClient &&
              (canBuyAllDomains ||
                (!canBuyAllDomains && !CurrentUserStore.isFreeMembership())) &&
              (showRegisterBtn ? (
                <div className="form-field">
                  <div className="field-title">
                    {__('Register A New Domain')}
                  </div>
                  {registerDomainContainer}
                </div>
              ) : (
                <a
                  onClick={this._onClickshowRegisterBtn.bind(this)}
                  href="javascript: void(0);">
                  {__('Register New Domain')}
                </a>
              ))}
            <div className="hr" />
          </div>
        )}
        {
          <PremiumFeature
            featureName="custom_domain"
            overlayStyle="fieldOverlay"
            title={__('Domain|Connect to a custom domain for this site.')}
            hint={__(
              'Upgrade your account to either Limited or Pro to access this feature!',
            )}
            source="cd">
            <div className="form-field custom-domain-field">
              <div className="field-title">
                {__('Custom Domain/Subdomain')}
                <StrikinglyOrSxl>
                  <span className="s-pro-icon">
                    {__('EditorSettings|Pro / Limited')}
                  </span>
                  <span className="s-pro-icon">
                    {__('EditorSettings|Biz / Pro / Limited')}
                  </span>
                </StrikinglyOrSxl>
              </div>
              {tct(
                __(
                  'EditorSettings|If you already own a domain, enter it here. Empty it to disconnect.',
                ),
                {
                  root: <div className="field-hint" />,
                },
              )}

              <div className="custom-domain-update">
                <MoreartyInput
                  type="text"
                  placeholder={__('e.g. www.mydomain.com')}
                  value={currentDomain}
                  onChange={this._onChangeCustomDomain}
                />
                <div
                  className="s-btn"
                  onClick={this._onSaveCustomDomain}
                  data-track-id="Update Custom Domain Button">
                  {__('Update')}
                  {['updating', 'checking'].includes(customDomainStatus) ? (
                    <i className="fa fa-spinner fa-pulse right-icon" />
                  ) : null}
                  {customDomainSaved ? <i className="fa fa-check" /> : null}
                </div>
                {ConfStore.getCanSeeDomainTransferTab() &&
                  !isSxl &&
                  hasNoDomainId &&
                  customDomainStatus !== 'updating' && (
                    <a
                      target="_blank"
                      href="/s#/v2_domains/transfer"
                      data-track-id="Click Transfer Domain Button">
                      {__('Transfer your domain to us!')}
                    </a>
                  )}
                {isSxl &&
                  currentDomain && (
                    <i
                      className="fa fa-qrcode qrcode-btn"
                      style={{ marginLeft: '10px' }}
                      onClick={() =>
                        this.props.showQrCode(`http://${currentDomain}`)
                      }
                    />
                  )}
                {showConnectedMessage && (
                  <div className="s-box-tag green">
                    {__('EditorSettings|Domain is connected!')}
                  </div>
                )}
                {isSxl &&
                  !this.props.customDomain && (
                    <div style={{ display: 'block' }}>
                      <ol className="connect-custom-domain-tips">
                        <li>
                          <Icon type="fa-check-circle-o" /> SEO 排名优化
                        </li>
                        <li>
                          <Icon type="fa-check-circle-o" /> 品牌更专业
                        </li>
                        <li>
                          <Icon type="fa-check-circle-o" /> 访问更快速
                        </li>
                      </ol>
                      <a href="https://wanwang.aliyun.com/" target="_blank">
                        还没有域名？前往万网购买
                      </a>
                    </div>
                  )}
              </div>

              {customDomainStatus !== 'updating' && domainSettingsRs.renderRs}

              {customDomainStatus === 'error' && (
                <div className="field-notice error">
                  <div>
                    <i className="entypo-info-circled" />
                    {customDomainMessage === 'Already taken'
                      ? tct(
                          __(
                            'Domain|Sorry,[placeholderStrong: %{domain}] is already taken.',
                            {
                              domain: this.state.currentDomain,
                            },
                          ),
                          {
                            root: <span />,
                            placeholderStrong: <strong />,
                          },
                        )
                      : customDomainMessage}
                  </div>
                </div>
              )}

              {!['updating', 'checking'].includes(customDomainStatus) &&
                customDomainCheckRs &&
                customDomainCheckRs.domainRegistered === false && (
                  <div className="field-notice error">
                    <div>
                      <i className="entypo-info-circled" />
                      {domainSupported ? (
                        <span>
                          {__(
                            'EditorSettings|This domain is not registered yet! You can register it now.',
                          )}
                          &nbsp;
                          <StrikinglyOrSxl>
                            <a
                              href="javascript: void(0);"
                              onClick={this._onRegisterNewDomainWithName}>
                              {__('EditorSettings|Claim your Domain')}
                            </a>
                            {null}
                          </StrikinglyOrSxl>
                        </span>
                      ) : (
                        <span>
                          {__(
                            'EditorSettings|This domain is not registered yet!',
                          )}
                        </span>
                      )}
                    </div>
                  </div>
                )}
            </div>
          </PremiumFeature>
        }

        {<div className="hr" />}

        <div className="form-field permalink">
          <div className="field-title">{__('Strikingly.com URL')}</div>
          <div className="field-hint">
            {__(
              'Enter a unique Strikingly.com URL. You may change this at any time.',
            )}
          </div>
          {
            <div>
              <div className="permalink-container">
                {permalinkInner}
                <div
                  className="s-btn permalink-update-btn"
                  onClick={this._onSavePermalink}>
                  {__('Update')}
                  {isUpdating ? <i className="fa fa-spinner fa-pulse" /> : null}
                  {isSaved ? <i className="fa fa-check" /> : null}
                </div>
                {isSxl &&
                  permalink && (
                    <i
                      className="fa fa-qrcode qrcode-btn"
                      onClick={() => this.props.showQrCode(completedPermalink)}
                    />
                  )}
              </div>
              <div className="tab-status">{status}</div>
            </div>
          }
        </div>
      </div>
    )
  }

  render() {
    const step = this.getDefaultBinding().get('currentStep')
    const domainPurchaseStatus = this.getDefaultBinding().get(
      'domainPurchaseStatus',
    )
    const v2DomainSetting = this.getDefaultBinding().get('v2DomainSetting')
    const { currentTabName, changeTabName } = this.props

    return (
      <div>
        {step === 'tab' ? (
          this._renderDomainTab()
        ) : (
          <PurchaseContainer
            key="second-step"
            inSettingsPanel={true}
            domainPurchaseStatus={domainPurchaseStatus}
            v2DomainSetting={v2DomainSetting}
            currentTabName={currentTabName}
            changeTabName={changeTabName}
            backToTab={this._onBackToTab}
            connectDomain={this._onConnectDomainAfterRegister}
            initialDomainName={this._initialDomainName}
          />
        )}
      </div>
    )
  }
}

/*
  Connected wrapper will use shadow equal for props checking by default.
  So when binding content changed, connected wrapper will avoid to render, binding ref is not changed (binding is a mutable object).
  Set pure option to false to let connected wrapper can re-render when it's parent calls re-render.

  The extra cost for this change is the re-render for connected wrapper (HOC).
  Won't introduce addtional cost (won't cause addtional render) for the original component (DomainsTab).
 */
export default connect(
  null,
  {
    showQrCode: url => openDialog('urlQrCodeDialog', { url }),
  },
  null,
  {
    pure: false,
  },
)(DomainsTab)
