let defaultExport = {}
if (__IN_EDITOR__ || __SERVER_RENDERING__) {
  class PageAnalyticsEngine {
    init() {}

    logPageView() {}
  }

  defaultExport = new PageAnalyticsEngine()
} else {
  const $ = require('jquery')
  const gaq = require('gaq')
  const Keen = require('js/vendor/keen-loader.js')
  const _ = require('lodash')

  const ConfStore = require('js/stores/conf_store')
  const PageMetaStore = require('js/stores/page_meta_store')
  const PageDataStore = require('js/stores/page_data_store')
  const _b = require('js/utils/lodashb')
  const logger = require('js/utils/logger')
  const ScriptHelper = require('js/utils/helpers/ScriptHelper')
  const VideoHelper = require('js/utils/helpers/video_helper')
  const ReferrerParser = require('js/utils/referrer_parser')

  const edit_page = require('js/v3_bridge/edit_page_bridge')
  const EcommerceGoogleAnalytics = require('js/v3_bridge/EcommerceGoogleAnalytics')

  const PageAnalyticsEngine = class PageAnalyticsEngine {
    constructor() {
      this.logPageView = this.logPageView.bind(this)
      this.logSocialClicks = this.logSocialClicks.bind(this)
      this.normalizedReferrer = this.normalizedReferrer.bind(this)
      this.sendPbsImpression = this.sendPbsImpression.bind(this)
      this.sendPbsConversion = this.sendPbsConversion.bind(this)
      this.sendDataKeenIO = this.sendDataKeenIO.bind(this)
    }

    init(data) {
      let isReturnVisit, visitorId
      if ($.cookie('__strk_visitor_id')) {
        visitorId = $.cookie('__strk_visitor_id')
        isReturnVisit = true
      } else {
        // a common way to generate a random id
        visitorId = 'visotor-xxxxxxxxxxxx4xxxyxxxxxxxxxxxxxxx'.replace(
          /[xy]/g,
          c => {
            const r = (Math.random() * 16) | 0
            const v = c === 'x' ? r : (r & 0x3) | 0x8
            return v.toString(16)
          },
        )
        isReturnVisit = false
        $.cookie('__strk_visitor_id', visitorId, { expires: 365 })
      }

      this.baseData = data || {
        pageId: PageMetaStore.getId(),
        userId: PageMetaStore.getUser().get('id'),
        permalink: PageMetaStore.getPermalink(),
        membership: PageMetaStore.getUser().get('membership'),
        createdAt: PageMetaStore.getCreatedAt(),
        strikinglyBranding: PageMetaStore.getShowStrikinglyLogo(),
      }

      this.baseData = _.extend(this.baseData, {
        visitorId,
        isReturnVisit,
        referrer: document.referrer,
      })

      if (ConfStore.getIsBlog()) {
        this._internals = {
          user: {
            id: PageMetaStore.getUserId(),
            membership: PageMetaStore.getMemberShip(),
          },
          page: {
            id: PageMetaStore.getId(),
            url() {
              return window.location.href
            },
            googleAnalyticsTracker: PageMetaStore.getGoogleAnalyticsTracker(),
            googleAnalyticsType: PageMetaStore.getGoogleAnalyticsType(),
            facebookPixelId: PageMetaStore.getFacebookPixelId(),
          },
        }
      } else {
        this._internals = {
          user: {
            id: PageMetaStore.getUser().get('id'),
            membership: PageMetaStore.getUser().get('membership'),
          },
          page: {
            id: PageMetaStore.getId(),
            url() {
              return window.location.href
            },
            theme: PageMetaStore.getTheme().get('name'),
            strk_upvt: PageMetaStore.getStrkUpvt(),
            strkGaTracker: PageMetaStore.getStrkGaTracker(),
            googleAnalyticsTracker: PageMetaStore.getGoogleAnalyticsTracker(),
            googleAnalyticsType: PageMetaStore.getGoogleAnalyticsType(),
            facebookPixelId: PageMetaStore.getFacebookPixelId(),
          },
        }
      }

      _b.traverseObj(this._internals, h => {
        for (const k in h) {
          const v = h[k]
          if (_.isUndefined(v)) {
            console.warn(`${k} is undefned`)
          }
        }
      })

      this.strikinglyGoogleAnalyticsEnabled = ConfStore.isGoogleAnalyticsEnabled()
      this.strikinglyKeenAnalyticsEnabled = ConfStore.isKeenAnalyticsEnabled()

      this.setupGA()
      return this.setupFB()
    }

    setSocialShareTracking() {
      // subscribe to facebook like event
      edit_page.Event.subscribe('Site.facebook.edge.create', () =>
        this.trackSocialMediaShare('facebook', 'like'),
      )
      // subscribe to linkedin share event
      edit_page.Event.subscribe('Site.linkedin.share', () =>
        this.trackSocialMediaShare('linkedin', 'share'),
      )
      // subscribe to twitter tweet event
      edit_page.Event.subscribe('Site.twitter.tweet', () =>
        this.trackSocialMediaShare('twitter', 'tweet'),
      )
      // subscribe to G+ +1 event
      return edit_page.Event.subscribe('Site.gplus.plusone', () =>
        this.trackSocialMediaShare('gplus', 'plusone'),
      )
    }

    userGoogleAnalyticsEnabled() {
      return Boolean(
        this._internals != null
          ? this._internals.page.googleAnalyticsTracker
          : undefined,
      )
    }

    userUniversalAnalyticsEnabled() {
      return this._internals.page.googleAnalyticsType === 'universal'
    }

    setupGA() {
      if (this.strikinglyGoogleAnalyticsEnabled) {
        logger.log(
          '[GA] Setup internal GA: ',
          this._internals.page.strkGaTracker,
        )
        if (typeof __ga === 'function') {
          __ga('create', this._internals.page.strkGaTracker, {
            name: 'strk',
            cookieDomain: 'auto',
          })
          __ga('strk.set', 'anonymizeIp', true)
        }
      }

      if (this.userGoogleAnalyticsEnabled()) {
        logger.log(
          '[GA] Setup GA for user: ',
          this._internals.page.googleAnalyticsTracker,
        )
        if (this.userUniversalAnalyticsEnabled()) {
          logger.log('[GA] Initialize Universal Analytics')
          window.ga = __ga
          if (typeof ga === 'function') {
            ga('create', this._internals.page.googleAnalyticsTracker, 'auto')
            ga('set', 'anonymizeIp', true)
          }
        } else {
          logger.log('[GA] Initialize Classic Analytics')
          gaq.push(['_setAccount', this._internals.page.googleAnalyticsTracker])
          gaq.push(['_gat._anonymizeIp'])
        }
      }
    }


    setupFB() {
      if (this._internals.page.facebookPixelId && !this._fbPixelLoaded) {
        if (!window._strk_fbq) {
          ScriptHelper.loadFB()
        }
        window._strk_fbq('init', this._internals.page.facebookPixelId)
        this._fbPixelLoaded = true
        return this._fbPixelLoaded
      }
    }

    // TODO
    // https://www.facebook.com/business/help/952192354843755

    trackPageViewOnGA(path) {
      if (this.strikinglyGoogleAnalyticsEnabled) {
        logger.log('[GA] Send page view to internal GA')
        if (typeof __ga === 'function') {
          __ga('strk.send', 'pageview', { 'anonymizeIp': true })
        }
      }

      if (this.userGoogleAnalyticsEnabled()) {
        if (path) {
          const query = location.search || ''
          path += query
        }
        if (this.userUniversalAnalyticsEnabled()) {
          logger.log('[GA] Send page view to user GA (Universal)')
          if (path) {
            if (typeof ga === 'function') {
              ga('set', 'page', path)
            }
          }
          if (typeof ga === 'function') {
            ga('send', 'pageview', { 'anonymizeIp': true })
          }
        } else {
          logger.log('[GA] Send page view to user GA (Classic)')
          const params = ['_trackPageview']
          if (path) {
            params.push(path)
          }
          gaq.push(params)
        }
      }
    }

    trackPageEventOnGA(
      category,
      action,
      label = null,
      value = null,
      eventData = {},
    ) {
      if (this.strikinglyGoogleAnalyticsEnabled) {
        logger.log(
          '[GA] Send page event to internal GA: ',
          category,
          action,
          label,
          value,
        )
        if (typeof __ga === 'function') {
          __ga('strk.send', 'event', category, action, label, value, { 'anonymizeIp': true })
        }
      }

      if (this.userGoogleAnalyticsEnabled()) {
        if (this.userUniversalAnalyticsEnabled()) {
          logger.log('[GA] Send page event to user GA (Universal)')
          if (typeof ga === 'function') {
            ga('send', 'event', category, action, label, value, { 'anonymizeIp': true })
          }
        } else {
          logger.log('[GA] Send page event to user GA (Classic)')
          let i = 1
          for (const k in eventData) {
            const v = eventData[k]
            gaq.push(['_setCustomVar', i, k, v, 3])
            ++i
          }
          gaq.push(['_trackEvent', category, action, label, value])
        }
      }
    }

    trackPageEventOnFB(name, options = {}) {
      if (window._strk_fbq) {
        window._strk_fbq('track', name, options)
      }
    }

    // TODO
    // https://www.facebook.com/business/help/952192354843755

    trackPageEvent() {
      const trackButton = (eventName, options) => {
        const that = this
        return function(event) {
          const t = $(this)

          const data = {
            url: t.attr('href'),
            target: t.attr('target'),
            text: t.text(),
          }

          edit_page.Event.publish(eventName, data)

          that.trackPageEventOnGA(
            'Action',
            options.gaEventName,
            data.text,
            null,
            {
              url: data.url,
              text: data.text,
            },
          )

          // no need to refresh page when link to section or page
          const hash =
            typeof data.url === 'string' && _.startsWith(data.url, '#')
          const pageLink =
            typeof data.url === 'string' &&
            (t[0].hostname === window.location.hostname ||
              _.startsWith(data.url, '/'))
          const isVideoThatOpensInOverlay = VideoHelper.needToTriggerHelper(
            data.url,
          )
          if (
            data.url &&
            data.target !== '_blank' &&
            !hash &&
            !pageLink &&
            !isVideoThatOpensInOverlay
          ) {
            event.preventDefault()
            setTimeout(() => (window.location.href = data.url), 500)
          }
        }
      }

      $('.s-button .s-common-button').click(
        trackButton('Site.button.click', { gaEventName: 'ButtonClick' }),
      )
    }

    trackSocialMediaShare(channel, action, data = null) {
      return this.trackUserPageEvent(
        ConfStore.getKeenIoPageSocialShareCollection(),
        {
          user: {
            id: this._internals.user.id,
            membership: this._internals.user.membership,
          },
          page: {
            id: this._internals.page.id,
            url: this._internals.page.url(),
            category: this._internals.page.category,
            theme: this._internals.page.theme,
          },
          channel,
          action,
          data,
        },
      )
    }

    trackPageFraming(domain) {
      return this.trackUserPageEvent(
        ConfStore.getKeenIoPageFramingCollection(),
        {
          user: {
            id: this._internals.user.id,
            membership: this._internals.user.membership,
          },
          page: {
            id: this._internals.page.id,
            category: this._internals.page.category,
          },
        },
      )
    }

    logPageView(extraEventData) {
      this.trackPageViewOnGA()
      this.trackPageEventOnFB('PageView')

      // fix for single page high bounce rate issue
      // http://aroundanalytics.com/trick-to-solve-high-bounce-rate-of-google-analytics-in-a-one-page-site/
      setTimeout(() => this.trackPageEventOnGA('Control', 'Bounce Rate'), 30000)

      return this.sendDataKeenIO(_.extend({}, this.baseData, extraEventData))
    }

    logSocialClicks(channel) {
      let data
      return (data = _.extend(
        { eventName: 'SocialClicks', channel },
        this.baseData,
      ))
    }

    normalizedReferrer(referrer_url) {
      // TODO: rewrite referrer parser
      const REFERRER_SOURCE = require('../data/referrer_source.json')
      const referrerParser = new ReferrerParser(REFERRER_SOURCE, referrer_url)

      // Don't use I18n here. We are writing data to keenio, use i18n here will
      // fuck up the data.
      return (
        (referrerParser.referrer != null
          ? referrerParser.referrer.name
          : undefined) ||
        referrerParser.url ||
        'Direct Traffic'
      )
    }

    sendPbsImpression(data) {
      if (!this.strikinglyKeenAnalyticsEnabled) {
        return
      }
      logger.log('[PBS] Impression', data)
      return Keen.addEvent(ConfStore.getKeenIoPbsImpressionCollection(), data)
    }

    sendPbsConversion(data) {
      if (!this.strikinglyKeenAnalyticsEnabled) {
        return
      }
      logger.log('[PBS] Conversion', data)
      return Keen.addEvent(ConfStore.getKeenIoPbsConversionCollection(), data)
    }

    trackUserPageEvent(collection, data) {
      if (!this.strikinglyKeenAnalyticsEnabled) {
        return
      }
      logger.log('User Page Event Tracking', collection, data)
      return Keen.addEvent(collection, data)
    }

    trackEcommerceBuyerEvent(name, data) {
      data = _.extend(
        {
          keen: {
            addons: [
              {
                name: 'keen:ip_to_geo',
                input: {
                  ip: 'ip_address',
                },
                output: 'ip_geo_info',
              },
              {
                name: 'keen:ua_parser',
                input: {
                  ua_string: 'user_agent',
                },
                output: 'parsed_user_agent',
              },
            ],
          },
          ip_address: '${keen.ip}',
          user_agent: '${keen.user_agent}',
          user: {
            id: this._internals.user.id,
            membership: this._internals.user.membership,
          },
          page: {
            id: this._internals.page.id,
            url: this._internals.page.url(),
            category: this._internals.page.category,
            theme: this._internals.page.theme,
          },
        },
        data,
      )

      return this.trackUserPageEvent(name, data)
    }

    sendDataKeenIO(data) {
      if (!this.strikinglyKeenAnalyticsEnabled) {
        return
      }
      const referrerHost = data.referrer.split('/')[2]
      let pageview = _.extend(
        {
          keen: {
            addons: [
              {
                name: 'keen:ip_to_geo',
                input: {
                  ip: 'ip_address',
                },
                output: 'ip_geo_info',
              },
              {
                name: 'keen:ua_parser',
                input: {
                  ua_string: 'user_agent',
                },
                output: 'parsed_user_agent',
              },
            ],
          },
          ip_address: '${keen.ip}',
          user_agent: '${keen.user_agent}',
          host: document.location.host,
          // parse the referrer url to get the host name of referrer
          referrer_host: referrerHost,
          normalized_referrer: this.normalizedReferrer(data.referrer),
        },
        data,
      )

      // For multi-page analytics dashboard
      // Never add these extra fields for blog
      if (!ConfStore.getIsBlog()) {
        pageview = _.extend(
          {
            is_multipage: PageDataStore.getIsMultiPage(),
            page_uid: PageDataStore.getCurrentPageUID(),
          },
          pageview,
        )
      }

      // Keen.addEvent($S.conf.keenio_collection_sharding, pageview)

      // TODO: delete it
      pageview.sharding = true
      return Keen.addEvent($S.conf.keenio_collection, pageview)
    }

    trackFileDownload(fileID) {
      if (!this.strikinglyKeenAnalyticsEnabled) {
        return
      }
      const trackData = {
        keen: {
          addons: [
            {
              name: 'keen:ip_to_geo',
              input: {
                ip: 'ip_address',
              },
              output: 'ip_geo_info',
            },
            {
              name: 'keen:ua_parser',
              input: {
                ua_string: 'user_agent',
              },
              output: 'parsed_user_agent',
            },
          ],
        },
        ip_address: '${keen.ip}',
        user_agent: '${keen.user_agent}',
        file_id: fileID,
        user: {
          id: this._internals.user.id,
          membership: this._internals.user.membership,
        },
        page: {
          id: this._internals.page.id,
          url: this._internals.page.url(),
          category: this._internals.page.category,
          theme: this._internals.page.theme,
        },
      }
      logger.log('File Download', trackData)
      return Keen.addEvent(
        ConfStore.getKeenIoFileDownloadCollection(),
        trackData,
      )
    }
  }
  PageAnalyticsEngine.prototype.pingInterval = 10000

  _.extend(PageAnalyticsEngine.prototype, EcommerceGoogleAnalytics)

  defaultExport = new PageAnalyticsEngine()
}
export default defaultExport
