import './init'
import _ from 'lodash'
import $ from 'jquery'
import 'js/vendor/jquery/easing'
import React from 'react'
import ReactDOM from 'react-dom'
import { Iterable } from 'immutable'

import { createStore, applyMiddleware } from 'redux'
import { createLogger } from 'redux-logger'
import { Provider } from 'react-redux'
import thunkMiddleware from 'redux-thunk'
import ManagerReducer from 'nextgen/ecommerce/manager/reducers/ManagerReducer'
import connectReduxStoreToBinding from 'js/utils/helpers/connectReduxStoreToBinding'

import PortfolioManagerStore from 'js/stores/PortfolioManagerStore'
import FeatureStore from 'js/stores/FeatureStore'
import ConfStore from 'js/stores/conf_store'

import * as UrlConstants from 'js/constants/url_constants'

import ProductPanel from 'nextgen/portfolio/manager/components/productPanel'
import CategoryManagerWrapper from 'nextgen/ecommerce/manager/components/settings/categoryManagerWrapper'

import gonReader from 'js/utils/gonReader'
import { AppContainer } from 'react-hot-loader'
import { ErrorBoundary } from 'js/components/ErrorBoundary'

import { UPDATE_SETTINGS_SUCCESS } from 'nextgen/ecommerce/manager/actions/entities/settings'

import PortfolioManagerBridge from 'js/v4_bridge/PortfolioManagerBridge'

import * as i18nHelper from 'js/utils/helpers/i18nHelper'

import ComponentKitContext from 'js/utils/ComponentKitContext'

const p1 = import(`locales/${i18nHelper.getTranslationFile()}`)

const loggerMiddleware = createLogger({
  collapsed: true,
  predicate: (/* getState, action */) => process.env.NODE_ENV !== 'production',
  stateTransformer: state => {
    if (Iterable.isIterable(state)) {
      return state.toJS()
    }
    return state
  },
})

const store = createStore(
  ManagerReducer,
  applyMiddleware(
    thunkMiddleware, // lets us dispatch() functions
    loggerMiddleware, // neat middleware that logs actions
  ),
)

const baseProps = {
  siteId: gonReader('id'),
  siteState: gonReader('state'),
  publicUrl: gonReader('public_url'),
  meta: gonReader('pageOwner.portfolio'),
  isStandAlone: window.parent === window,
  isPro: ['pro', 'namecheap', 'sxlbiz'].includes(
    gonReader('pageOwner.membership'),
  ),
  isVip: gonReader('pageOwner.membership') === 'vip',
  isSxl: Boolean(gonReader('globalConf.isSxl')),
  isAdmin: Boolean(gonReader('pageOwner.isAdmin')),
  isSupport: Boolean(gonReader('pageOwner.isSupport')),
}

const productProps = {
  ...baseProps,
  productLimit:
    gonReader('pageOwner.portfolio').productLimit ||
    (gonReader('globalConf.isSxl') ? 6 : 1),
  upgradeUrl: UrlConstants.PRICING.GOTO_AND_RETURN(
    gonReader('id'),
    'portfolio',
  ),
  productPageRollout: gonReader('globalConf.rollout.product_page'),
  productDetailRollout: gonReader('globalConf.rollout.product_detail'),
  siteMemberShip: gonReader('globalConf.rollout.siteMemberShip'),
  noCategoryLimit: gonReader('globalConf.rollout.no_category_limit'),
}

class ProductsContainer extends React.Component {
  componentWillMount() {
    PortfolioManagerBridge.subPublishUrlChange((topic, data) => {
      baseProps.publicUrl = data.url
      this.forceUpdate()
    })

    PortfolioManagerBridge.subSiteStateChange((topic, data) => {
      baseProps.siteState = data.state
      this.forceUpdate()
    })
  }

  render() {
    return (
      <Provider store={store}>
        <ProductPanel
          {...productProps}
          canUseCategory={FeatureStore.canUse('portfolio_category')}
          canSeeCategory={FeatureStore.canSee('portfolio_category')}
        />
      </Provider>
    )
  }
}

function CategoryContainer(/* props */) {
  return (
    <Provider store={store}>
      <CategoryManagerWrapper
        {...productProps}
        isPortfolio={true}
        canUseCategory={FeatureStore.canUse('portfolio_category')}
        canSeeCategory={FeatureStore.canSee('portfolio_category')}
      />
    </Provider>
  )
}

const CategoryContainerWithComponentKit = ComponentKitContext(CategoryContainer)

const siteIdString = gonReader('id').toString()
PortfolioManagerBridge.subSettingsChange((topic, data) => {
  let settings =
    store
      .getState()
      .getIn(['entities', 'settings', 'data', siteIdString, 'settings']) || {}
  if (settings.toJS) {
    settings = settings.toJS()
  }

  store.dispatch({
    type: UPDATE_SETTINGS_SUCCESS,
    payload: {
      settings: Object.assign(settings, data.settings),
    },
    meta: {
      siteId: siteIdString,
    },
  })
})

const getState = () => ({
  product: store
    .getState()
    .getIn(['entities', 'product', 'data', siteIdString]),
  category: store.getState().getIn(['entities', 'category', 'data']),
  settings: store
    .getState()
    .getIn(['entities', 'settings', 'data', siteIdString, 'settings']),
})

const prevState = getState()
store.subscribe(() => {
  const newState = getState()
  if (newState.product !== prevState.product) {
    prevState.product = newState.product
    PortfolioManagerBridge.pubProductsChange(
      _.flatten(_.toArray(newState.product.toJS())),
    )
  }

  if (newState.settings !== prevState.settings) {
    prevState.settings = newState.settings
    PortfolioManagerBridge.pubSettingsChange(newState.settings.toJS())
  }

  if (newState.category !== prevState.category) {
    prevState.category = newState.category
    PortfolioManagerBridge.pubCategoriesChange(
      _.flatten(_.toArray(newState.category.toJS())),
    )
  }
})

Promise.all([p1])
  .then(([poFile]) => {
    const I18n = require('js/utils/i18n')
    I18n.init(poFile)
    $(() => {
      const cloudinary = require('cloudinary')
      cloudinary.config('cloud_name', ConfStore.getCloudinaryCloudName())

      if (window.parent === window) {
        let ImageAssetDialog = require('js/v4_bridge/react_app_bridge/ImageAssetDialogApp')
        const EditorStore = require('js/stores/editor_store')
        const ctx = EditorStore.init()
        connectReduxStoreToBinding(store, ctx.getBinding())
        ImageAssetDialog = ctx.bootstrap(ImageAssetDialog())
        ReactDOM.render(
          <Provider store={store}>
            <ErrorBoundary>
              <AppContainer>
                <ImageAssetDialog />
              </AppContainer>
            </ErrorBoundary>
          </Provider>,
          document.getElementById('container'),
        )
      }

      PortfolioManagerStore.init()
      FeatureStore.hydrate($S.features)

      const productsContext = PortfolioManagerStore.getProductsContext()
      const WrapperedProductsContainer = productsContext.bootstrap(
        ComponentKitContext(ProductsContainer),
      )
      ReactDOM.render(
        <AppContainer>
          <ErrorBoundary>
            <WrapperedProductsContainer />
          </ErrorBoundary>
        </AppContainer>,
        document.getElementById('items-container'),
      )
      ReactDOM.render(
        <ErrorBoundary>
          <CategoryContainerWithComponentKit />
        </ErrorBoundary>,
        document.getElementById('category-manager'),
      )
    })
  })
  .catch(e => console.error(e))

if (module.hot) {
  module.hot.accept(err => {
    if (err) {
      console.error(err)
    }
  })
}
