import { doGet, doPost, doPut } from 'wechat_common/utils/request'
import { PRESENTATION_API } from 'root/constants/presentation/urlConstants'
import compose from 'ramda/src/compose'
import { bindSiteId, bindCode } from 'wechat_common/utils/context'
import { bindRetry, bindShowToast } from 'wechat_common/networkErrorManager'

function fetchSettings(options) {
  return doGet(bindRetry(options))
}

function fetchPortfolioSettings(options) {
  return doGet(bindRetry(options))
}

function getProducts(url) {
  return doGet(bindRetry({ url }))
}

function getCategories(options) {
  return doGet(bindRetry(options))
}

function getProduct(options) {
  return doGet(bindRetry(options))
}

function getCode(options) {
  return doPost(bindRetry(options))
}

function verify(options) {
  return doPost(bindRetry(options))
}

function registerTeamMember(options) {
  return doPost(bindRetry(options))
}

export const fetchGroups = compose(bindSiteId, bindCode)((code, siteId) =>
  doGet({
    url: PRESENTATION_API.FETCH_GROUP_LIST(siteId, code),
  }),
)

export const fetchGroupSummaryDataById = compose(bindSiteId, bindCode)(
  (code, siteId, teamId) =>
    doGet({
      url: PRESENTATION_API.FETCH_GROUP_SUMMARY_DATA_BY_DI(
        siteId,
        code,
        teamId,
      ),
    }),
)

function createFeedback(options) {
  return doPost(options)
}

export const fetchGroupMembers = compose(bindSiteId, bindCode)(
  (code, siteId, groupId, page, rankingType) =>
    doGet({
      url: PRESENTATION_API.FETCH_GROUP_MEMBERS(
        siteId,
        code,
        groupId,
        page,
        rankingType,
      ),
    }),
)

export const bindCustomerRelationship = compose(bindSiteId, bindCode)(
  (code, siteId, teamMemberId) =>
    doPost(
      bindRetry({
        url: PRESENTATION_API.BIND_CUSTOMER_RELATIONSHIP(siteId),
        data: {
          code,
          teamMemberId,
        },
      }),
    ),
)

export const getTeamMemberInfoById = compose(bindSiteId, bindCode)(
  (code, siteId, teamMemberId) =>
    doGet(
      bindRetry({
        url: PRESENTATION_API.FETCH_TEAM_MEMBER_CARD_INFO(
          siteId,
          code,
          teamMemberId,
        ),
      }),
    ),
)

export const fetchTeamMemberCards = compose(bindSiteId, bindCode)(
  (code, siteId) =>
    doGet(
      bindRetry({
        url: PRESENTATION_API.FETCH_TEAM_MEMBER_CARDS(siteId, code),
      }),
    ),
)

export const updateCustomerRelationship = compose(bindSiteId, bindCode)(
  (code, siteId, currentTeamMemberId, targetTeamMemberId) =>
    doPost(
      bindShowToast({
        url: PRESENTATION_API.UPDATE_CUSTOMER_RELATIONSHIP(siteId),
        data: {
          memberId: currentTeamMemberId,
          targetTeamMemberId,
          code,
        },
      }),
    ),
)

export const fetchRadarUnreadCount = compose(bindSiteId, bindCode)(
  (code, siteId) =>
    doGet(
      bindRetry({
        url: PRESENTATION_API.FETCH_RADAR_UNREAD_COUNT(siteId, code),
      }),
    ),
)

export const fetchGroupChartAnalytics = compose(bindSiteId, bindCode)(
  (code, siteId, teamId, startDate, endDate) =>
    doGet({
      url: PRESENTATION_API.FETCH_GROUP_CHART_ANALYTICS(
        siteId,
        code,
        teamId,
        startDate,
        endDate,
      ),
    }),
)

const fetchGroupAnalyticsReports = compose(bindSiteId, bindCode)(
  (code, siteId, teamId, startDate, endDate) =>
    doGet({
      url: PRESENTATION_API.FETCH_GROUP_ANALYTICS_REPORTS(
        siteId,
        code,
        teamId,
        startDate,
        endDate,
      ),
    }),
)

const fetchMemberList = compose(bindSiteId, bindCode)(
  (code, siteId, value, memberId, groupId, needFakeData) =>
    doGet(
      bindRetry({
        url: PRESENTATION_API.FETCH_MEMBER_LIST(
          code,
          siteId,
          value,
          memberId,
          groupId,
          needFakeData,
        ),
      }),
    ),
)

const fetchVisitorDetail = compose(bindCode, bindSiteId)(
  (siteId, code, memberId, startDate, endDate, teamMemberId) =>
    doGet(
      bindRetry({
        url: PRESENTATION_API.FETCH_VISITOR_DETAIL(
          siteId,
          code,
          memberId,
          startDate,
          endDate,
          teamMemberId,
        ),
      }),
    ),
)

const fetchVisitorInfo = compose(bindSiteId)((siteId, memberId) =>
  doGet(
    bindRetry({
      url: PRESENTATION_API.FETCH_VISITOR_INFO(siteId, memberId),
    }),
  ),
)

const fetchVisitorAnalytics = compose(bindSiteId)((siteId, memberId, options) =>
  doGet({
    url: PRESENTATION_API.FETCH_VISITOR_ANALYTICS(siteId, memberId, options),
  }),
)

const fetchVisitorGeneralReport = compose(bindSiteId)(
  (siteId, memberId, options) =>
    doGet({
      url: PRESENTATION_API.FETCH_VISITOR_GENERAL_REPORT(
        siteId,
        memberId,
        options,
      ),
    }),
)

const updateVisitorInfo = compose(bindSiteId)(
  (siteId, code, memberId, contactInfo) =>
    doPut(
      bindShowToast({
        url: PRESENTATION_API.UPDATE_VISITOR_INFO(siteId, memberId),
        data: {
          code,
          contactInfo,
        },
      }),
    ),
)

const fetchContactForms = compose(bindSiteId, bindCode)((code, siteId) =>
  doGet(
    bindRetry({
      url: PRESENTATION_API.FETCH_CONTACT_FORMS(siteId, code),
    }),
  ),
)

const updateContactFormsReadMark = compose(bindSiteId, bindCode)(
  (code, siteId, messageId) =>
    doPost({
      url: PRESENTATION_API.UPDATE_CONTACT_FORMS_READ_MARK(siteId),
      data: {
        code,
        contact_form_ids: [messageId],
      },
    }),
)

function createContactForm(options) {
  return doPost(options)
}

function fetchFormSettings(options) {
  return doGet(bindRetry(options))
}

function getShare(options) {
  return doPost(bindRetry(options))
}

const updateMemberCardInfo = compose(bindSiteId, bindCode)(
  (code, siteId, value, memberId) =>
    doPut(
      bindShowToast({
        url: PRESENTATION_API.UPDATE_MEMBER_CARD_INFO(siteId, memberId),
        data: Object.assign({}, {code}, value),
      }),
    ),
)

const updateTeamMember = compose(bindCode, bindSiteId)(
  (siteId, code, memberId, data) =>
    doPut(
      bindShowToast({
        url: PRESENTATION_API.UPDATE_TEAM_MEMBER(siteId, memberId, code),
        data,
      }),
    ),
)

const updateMemberLikes = compose(bindSiteId, bindCode)(
  (code, siteId, likeState, teamMemberId) =>
    doPut(
      bindShowToast({
        url: PRESENTATION_API.UPDATE_MEMBER_LIKES(siteId),
        data: {
          code,
          likeState,
          teamMemberId,
        },
      }),
    ),
)

const getChannelList = compose(bindCode, bindSiteId)(
  (siteId, code, memberId, groupId) => {
    const url = PRESENTATION_API.GET_CHANNEL_LIST(
      siteId,
      code,
      memberId,
      groupId,
    )

    return doGet(
      bindRetry({
        url,
      }),
    )
  },
)

const getChannelDetail = compose(bindCode, bindSiteId)(
  (siteId, code, channelId, memberId, groupId, serviceNotification) => {
    const url = PRESENTATION_API.GET_CHANNEL_DETAIL(
      siteId,
      code,
      channelId,
      memberId,
      groupId,
      serviceNotification,
    )
    return doGet({
      url,
    })
  },
)

const getChannelUnreadCount = compose(bindCode, bindSiteId)(
  (siteId, code, channelId) => {
    const url = PRESENTATION_API.GET_CHANNEL_UNREAD(siteId, code, channelId)

    return doGet({
      url,
    })
  },
)

const postChannelDetail = compose(bindCode, bindSiteId)(
  (siteId, code, channelId, data) => {
    const url = PRESENTATION_API.POST_CHANNEL_DETAIL(siteId, code)

    data.channelId = channelId

    return doPost({
      url,
      data,
    })
  },
)

const getBroadcasts = compose(bindCode, bindSiteId)((siteId, code) => {
  const url = PRESENTATION_API.GET_BROADCASTS(siteId, code)
  return doGet(
    bindRetry({
      url,
    }),
  )
})

const getSystemNotifications = compose(bindSiteId)(
  (siteId, teamMemberId, startDate, endDate) => {
    const url = PRESENTATION_API.GET_SYSTEM_NOTIFICATIONS(
      siteId,
      teamMemberId,
      startDate,
      endDate,
    )
    return doGet(
      bindRetry({
        url,
      }),
    )
  },
)

const createBroadcast = compose(bindCode, bindSiteId)((siteId, code, data) => {
  const url = PRESENTATION_API.CREATE_BROADCAST(siteId, code)

  return doPost({
    url,
    data,
  })
})

const getChannelListUnread = compose(bindCode, bindSiteId)((siteId, code) => {
  const url = PRESENTATION_API.GET_CHANNEL_LIST_UNREAD(siteId, code)

  return doGet(
    bindRetry({
      url,
    }),
  )
})

const fetchTrackEvents = compose(bindCode, bindSiteId)(
  (siteId, code, startDate, endDate) => {
    const url = PRESENTATION_API.FETCH_TRACK_EVENTS(
      siteId,
      code,
      startDate,
      endDate,
    )
    return doGet(
      bindRetry({
        url,
      }),
    )
  },
)

const fetchProductDetail = compose(bindSiteId)((siteId, productId) => {
  const url = PRESENTATION_API.GET_PRODUCT_DETAIL(siteId, productId)
  return doGet({
    url,
  })
})

const fetchAnalyticsReport = compose(bindCode, bindSiteId)(
  (siteId, code, teamMemberId, startDate, endDate, interval) => {
    const url = PRESENTATION_API.FETCH_ANALYTICS_REPORTS(
      siteId,
      code,
      teamMemberId,
      startDate,
      endDate,
      interval,
    )
    return doGet({
      url,
    })
  },
)

const getServiceMessageInfo = compose(bindSiteId)(
  (siteId, clientId, teamMemberId) => {
    const url = PRESENTATION_API.GET_SERVICE_MESSAGE_INFO(
      siteId,
      clientId,
      teamMemberId,
    )
    return doGet({
      url,
    })
  },
)

const fetchProductShare = compose(bindSiteId)(
  (siteId, productId, page, affiliateOpenId) => {
    const url = PRESENTATION_API.GET_PRODUCT_SHARE(
      siteId,
      productId,
      page,
      affiliateOpenId,
    )
    return doPost({
      url,
    })
  },
)

const fetchAppList = code => {
  const url = PRESENTATION_API.GET_APP_LIST(code)
  return doGet(bindRetry({ url }))
}

const postFormId = compose(bindCode, bindSiteId)((siteId, code, formId) => {
  const url = PRESENTATION_API.POST_VALID_FORM_ID(siteId)
  const data = {
    code,
    formId,
  }
  return doPost({
    url,
    data,
  })
})

const sendServiceMessage = compose(bindCode, bindSiteId)(
  (siteId, code, clientid, type) => {
    const url = PRESENTATION_API.POST_SEND_SERVICE(siteId)
    const data = {
      code,
      stCoreClientId: Number(clientid),
      notificationType: type,
    }
    return doPost({
      url,
      data,
    })
  },
)

const putServiceCount = compose(bindSiteId)((siteId, memberId) => {
  const url = PRESENTATION_API.PUT_SERVICE_NOTIFICATION_COUNT(siteId)
  const data = {
    member_id: memberId,
  }
  return doPut({
    url,
    data,
  })
})

const getServiceNotifyPopup = compose(bindCode, bindSiteId)((siteId, code) => {
  const url = PRESENTATION_API.FETCH_SERVICE_POPUP_STATUS(siteId, code)
  return doGet({
    url,
  })
})

const presentationApi = {
  fetchAnalyticsReport,
  fetchAppList,
  fetchSettings,
  getProducts,
  fetchPortfolioSettings,
  getCategories,
  getProduct,
  getCode,
  verify,
  registerTeamMember,
  fetchGroupMembers,
  fetchGroups,
  createFeedback,
  createContactForm,
  fetchFormSettings,
  fetchMemberList,
  fetchVisitorDetail,
  updateVisitorInfo,
  fetchVisitorInfo,
  fetchVisitorAnalytics,
  fetchVisitorGeneralReport,
  fetchContactForms,
  updateContactFormsReadMark,
  updateMemberCardInfo,
  updateTeamMember,
  updateMemberLikes,
  getChannelList,
  getChannelDetail,
  getChannelUnreadCount,
  postChannelDetail,
  getChannelListUnread,
  getBroadcasts,
  createBroadcast,
  fetchTrackEvents,
  fetchProductDetail,
  fetchProductShare,
  getShare,
  getSystemNotifications,
  fetchRadarUnreadCount,
  fetchGroupChartAnalytics,
  fetchGroupAnalyticsReports,
  fetchGroupSummaryDataById,
  postFormId,
  getServiceMessageInfo,
  sendServiceMessage,
  putServiceCount,
  getServiceNotifyPopup,
}

export default presentationApi
