#!/usr/bin/env node
const inquirer = require('inquirer')
const program = require('commander')
const { log, colors } = require('../packages/utils/log')
const { spawn } = require('child_process')
const _ = require('lodash')

const {
  updateConfigFiles,
  updateLegacyCodes,
  upgradePackages,
  preserveOldFiles,
  restoreOldFiles
} = require('../packages/actions')

const BANNER = `
             *                      *
                     .
          |\\___/|
          )     (             .              *
         =\\     /=
           )===(       *
          /     \\
          |     |
         /       \\
         \\       /
  _/\\_/\\_/\\__  _/_/\\_/\\_/\\_/\\_/\\_/\\_/\\_/\\_/\\_
  |  |  |  |( (  |  |  |  |  |  |  |  |  |  |
  |  |  |  | ) ) |  |  |  |  |  |  |  |  |  |
  |  |  |  |(_(  |  |  |  |  |  |  |  |  |  |
  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |
`

const VERSION = '0.1.0'

const run = async () => {
  console.info(BANNER)

  const { mode } = await inquirer.prompt([
    {
      name: 'mode',
      type: 'list',
      message: [
        colors.title('Which mode would like to choose:'),
        colors.item(
          '  🚝  chill - webpack 4, ES2018 output, cheap-eval-source-map'
        ),
        colors.item('  🚀  mad max - coming soon ...\n'), // add extra \n for good looking
        // '  🚀  mad max - webpack 4, ES2018 output, no source map, no backend log',
        ''
      ].join('\n'),
      // choices: ['chill', 'mad max']
      choices: ['chill']
    }
  ])

  // add extra \n for good looking
  console.log('\n')
  log.title('📥  Preserve current configs')
  preserveOldFiles()
  log.title('📝  Update config files')
  updateConfigFiles(mode)
  log.title('💯  Update legacy codes')
  updateLegacyCodes()
  log.title('🐣  Upgrade pacakages')
  upgradePackages()

  log.title('🐥  Everything set! Run yarn dev')
  // since shelljs.exec doesn't work with inqurier.js
  // use spawn instead
  spawn('yarn', ['dev'], {
    stdio: 'inherit'
  })
}

program.version(VERSION)

program
  .command('run')
  .description('Run wildcat')
  .action(run)

program
  .command('restore')
  .description(
    'Restore the previous configs, make sure you have execed wildcat run command or it may go wrong.'
  )
  .action(restoreOldFiles)

program.parse(process.argv)

// default command when no params attached
if (_.isEmpty(program.args)) {
  console.log(BANNER)
  colors.title(`Version: v${VERSION}\n`)
  console.log(
    `  ${colors.title(
      'Wildcat is a cli tool for accelerating webpack dev server of Bobcat.'
    )}`
  )
  program.help()
}
