// const path = require('path')
const fs = require('fs')
const path = require('path')
const chalk = require('chalk')
const shell = require('shelljs')
const handlebars = require('handlebars')

const { log } = require('./utils/log')

const { TEMPLATES, PACKAGES, LEGACY_CODES, BACKUP_FILES } = require('./configs')

handlebars.registerHelper('if_eq', function(a, b, opts) {
  return a === b ? opts.fn(this) : opts.inverse(this)
})

const preserveOldFiles = () => {
  BACKUP_FILES.forEach(file => {
    log.item(`Preserving ${file} ...`)
    const basename = path.basename(file)
    fs.copyFileSync(file, `${__dirname}/backups/${basename}`)
  })
}

const restoreOldFiles = () => {
  log.title('📤  Restore previous configs')
  log.title(
    `⚠️  Make sure you have execed ${chalk.bgGreen.black(
      'wildcat run'
    )}, or it may go wrong`
  )
  BACKUP_FILES.forEach(file => {
    log.item(`Restoring ${file} ...`)
    const basename = path.basename(file)
    fs.copyFileSync(`${__dirname}/backups/${basename}`, file)
  })
  log.title(
    `⚠️  Run ${chalk.bgGreen.black(
      'npm update'
    )} yourself to reset node_mouduls`
  )
}

const updateConfigFiles = (mode = 'chill') => {
  TEMPLATES.forEach(filename => {
    log.item(`Updating ${filename} ...`)
    const content = fs.readFileSync(
      `${__dirname}/templates/${filename}.template`,
      'utf8'
    )
    const template = handlebars.compile(content)
    const output = template({ mode })
    fs.writeFileSync(filename, output, 'utf8')
  })
}

const updateLegacyCodes = () => {
  LEGACY_CODES.forEach(({ name, update }) => {
    log.item(`Updating ${name} ...`)
    const content = fs.readFileSync(name, 'utf8')
    const output = update(content)
    fs.writeFileSync(name, output, 'utf8')
  })
}

const upgradePackages = () => {
  const command =
    'yarn add -D ' +
    PACKAGES.map(({ name, version }) => {
      log.item(`${name}:⛳️  ${chalk.bgGreen.black(version)}`)
      return `${name}@${version}`
    }).join(' ')
  const { code, stderr } = shell.exec(command)
  if (!code) console.error(stderr)
}

module.exports = {
  preserveOldFiles,
  restoreOldFiles,
  updateConfigFiles,
  updateLegacyCodes,
  upgradePackages
}
