// SPEC
// Owner: Dafeng
// Feature name: Font store
// * user can update fonts from choosing style
// NOTE:
// Specs here are all technical implementation. Will be ignored for the current QA.

import _ from 'lodash'
import Immutable from 'immutable'
import { EventEmitter } from 'events'

import EditorDispatcher from 'js/dispatcher/editor_dispatcher'
import EditorConstants from 'js/constants/editor_constants'
import BindingHelper from 'js/utils/helpers/binding_helper'
import weakmapMemoize from 'js/utils/weakmapMemoize'



const getDefaultFontName = require('js/utils/getDefaultFontName').default

let _bHelper
let _pDataHelper // need the binding to pageData to update fonts

const _updateFont = function(fontType, v) {
  const name = `${fontType}Font`
  _pDataHelper.setData(name, v)
  _clearPreviewFont()
}

const _updateFontPreset = function(preset) {
  _pDataHelper.setData('fontPreset', preset.id)
  ;['title', 'heading', 'body'].map(textType =>
    _updateFont(textType, preset.fonts[textType]),
  )
}

// No longer update fonts from style!
// _updateFontsFromStyle = (styles) ->
//   _.map ['heading','body','title'], (font) =>
//     _updateFont(font, styles[font + "Font"])

const _updatePreviewFont = (fontType, v) =>
  _bHelper.setData(`preview.${fontType}Font`, v)

const _updatePreviewFontPreset = preset =>
  ['title', 'heading', 'body'].map(textType =>
    _updatePreviewFont(textType, preset.fonts[textType]),
  )

const _clearPreviewFont = function() {
  const preview = _bHelper.binding.sub('preview')
  const transaction = preview.atomically()
  const object = preview.get().toJS()
  for (const key in object) {
    const v = object[key]
    transaction.set(key, undefined)
  }
  transaction.commit()
}

const FontStore = _.assign({}, EventEmitter.prototype, {
  // fontPreset is currently for Shangxianle, which designates font setting of
  // title, heading, and body

  // TODO - Andy, added 22 March 2016
  // Improve the font determination logic:
  // - first check if fontPreset is present, if so, use the preset setting
  // - if no fontPreset, use setting for titleFont, bodyFont etc.
  // Currently, fontPreset is just a collection of setting of the other three
  // fonts, which means, if we modify the style setting of a preset, the pages
  // that use this preset will NOT get style updates, unless they choose another
  // preset and switch back.

  _allFonts: null,
  _initialFonts: null,
  _fontsSelectedOnPageLoad: [],

  getDefault(pageData) {
    return {
      preview: {
        titleFont: '',
        bodyFont: '',
        headingFont: '',
        fontPreset: '',
      },
      data: {
        // use for notifying rich text component
        titleFont: pageData.titleFont,
        bodyFont: pageData.bodyFont,
        headingFont: pageData.headingFont,
        fontPreset: pageData.fontPreset,
      },
    }
  },

  init(b, pDataB) {
    _bHelper = new BindingHelper(b)
    const ConfStore = require('js/stores/conf_store')
    if (__IN_EDITOR__ && !ConfStore.getIsBlog()) {
      _pDataHelper = new BindingHelper(pDataB)

      // Manually synchronize side_menu and page_data
      // to avoid unnecessary render for side menu
      ;['headingFont', 'bodyFont', 'titleFont', 'fontPreset'].map(font =>
        _pDataHelper.binding.addListener(font, changes =>
          _bHelper.setData(`data.${font}`, _pDataHelper.getData(font)),
        ),
      )
    }
    return _bHelper.binding
  },

  _setBHelperForTests(bHelper) {
    return (_bHelper = bHelper)
  },

  loadFontsIfNotLoaded() {
    if (_bHelper.getData('isLoadingFonts')) {
      return
    }
    _bHelper.setData('isLoadingFonts', true)
    return import('../../../config/fonts.json')
      .then(fonts => {
        this._setAllFonts(fonts)
        return _bHelper.setData('isLoadingFonts', false)
      })
      .catch(() => _bHelper.setData('isLoadingFonts', false))
  },

  hydrate(fonts, pageData, initFonts) {
    this._initialFonts = initFonts
    _bHelper.binding
      .atomically()
      .set(Immutable.fromJS(this.getDefault(pageData)))
      .commit({ notify: false })
    return (this._fontsSelectedOnPageLoad = this._getUsedFonts())
  },

  getData(k) {
    return _bHelper.binding.get(k)
  },

  getBinding() {
    return _bHelper.binding
  },

  getFontName(type, options = {}) {
    let preview = false
    if (options.preview != null) {
      ;({ preview } = options)
    }
    if (preview) {
      return _bHelper.getData(`preview.${type}Font`)
    } else {
      return _bHelper.getData(`data.${type}Font`)
    }
  },

  getAvailableFonts() {
    if (this._allFonts) {
      return this._allFonts
    } else {
      return this._initialFonts
    }
  },

  search(fontUsage, searchTerm) {
    let matchAtBeginning = []
    let generalMatch = []

    const normalizeFontName = name => name.toLowerCase().replace(/ /g, '')
    searchTerm = normalizeFontName(searchTerm)

    this.getAvailableFonts().forEach(f => {
      if (fontUsage === 'body' && f.disableBody) {
        return
      }
      if (f.hidden) {
        return
      }
      const name = normalizeFontName(f.displayName)
      if (name.slice(0, searchTerm.length) === searchTerm) {
        return matchAtBeginning.push(f)
      } else if (name.indexOf(searchTerm) !== -1) {
        return generalMatch.push(f)
      }
    })

    matchAtBeginning = _.sortBy(matchAtBeginning, f => f.name)
    generalMatch = _.sortBy(generalMatch, f => f.name)
    return matchAtBeginning.concat(generalMatch).slice(0, 20)
  },

  _getSuggestedFonts() {
    return this.getVisibleFonts().filter(f => f.isSuggested)
  },

  _getUsedFonts() {
    return _([
      this.getFontName('title'),
      this.getFontName('heading'),
      this.getFontName('body'),
    ])
      .compact()
      .uniq()
      .map(name => this.getFontByName(name))
      .value()
  },

  getSuggestedFonts(fontUsage) {
    const usedFontsForOtherTextTypes = this._getUsedFonts()

    const defaultFontName = getDefaultFontName(fontUsage)
    const defaultFont = this.getFontByName(defaultFontName)

    let popular = this._getSuggestedFonts().concat(
      this._fontsSelectedOnPageLoad,
    )
    popular = _(popular)
      .filter(font => font.name !== defaultFontName)
      .sortBy(font => font.name)
      .value()

    let fonts = usedFontsForOtherTextTypes.concat([defaultFont]).concat(popular)
    fonts = _(fonts)
      .reject(f => {
        if (fontUsage === 'body' && f.disableBody) {
          return true
        }
        if (f.hidden) {
          return true
        }
      })
      .uniq(f => f.name)
      .value()

    return fonts
  },

  _setAllFonts(fonts) {
    this._allFonts = fonts
  },

  _getVisibleFonts: weakmapMemoize(availableFonts =>
    availableFonts.filter(f => !f.hidden),
  ),

  getVisibleFonts() {
    return this._getVisibleFonts(this.getAvailableFonts())
  },

  getTitleFonts() {
    return this.getVisibleFonts()
  },

  getHeadingFonts() {
    return this.getVisibleFonts()
  },

  _getBodyFonts: weakmapMemoize(visibleFonts =>
    _.select(visibleFonts, f => !f.disableBody),
  ),

  getBodyFonts() {
    return this._getBodyFonts(this.getVisibleFonts())
  },

  getFontByName(name) {
    return _.find(
      this.getAvailableFonts(),
      f => f.name.toLowerCase() === name.toLowerCase(),
    )
  },

  getFont(textType, options) {
    return _.find(
      this.getAvailableFonts(),
      item => item.name === this.getFontName(textType, options),
    )
  },

  getSelectedFontPresetName() {
    return _pDataHelper.getData('fontPreset')
  },

  // Look into preview fonts and actual font to determine which font to show
  // 1. if preview has value 'default', use theme default
  // 2. if preview has other values, use that
  // 3. otherwise, use font in page data
  getFontStyle(textType) {
    const previewFontName = this.getFontName(textType, { preview: true })
    if (previewFontName === 'default') {
      // fallback to theme defaults
      return {}
    } else {
      const font =
        this.getFont(textType, { preview: true }) ||
        this.getFont(textType, { preview: false })
      if (font != null) {
        return {
          fontFamily: font.cssValue,
        }
      } else {
        return {}
      }
    }
  },

  getFontClassNames() {
    // Class names on #s-content, used for both blog post and site in order to set font
    const fontClassNames = ['heading', 'title', 'body'].map(type => {
      const fontName =
        this.getFontName(type, { preview: true }) ||
        this.getFontName(type, { preview: false })
      const slug = fontName ? fontName.toSlug() : undefined
      if (slug) {
        return `s-font-${type}-${slug}`
      } else {
        return `s-font-${type}-default`
      }
    })

    return fontClassNames.join(' ')
  },
})

EditorDispatcher.register(payload => {
  switch (payload.actionType) {
    case EditorConstants.ActionTypes.SELECT_FONT:
      _updateFont(payload.fontType, payload.value)
      break
    case EditorConstants.ActionTypes.SELECT_FONT_PRESET:
      _updateFontPreset(payload.preset)
      break
    case EditorConstants.ActionTypes.PREVIEW_FONT:
      _updatePreviewFont(payload.fontType, payload.value)
      break
    case EditorConstants.ActionTypes.PREVIEW_FONT_PRESET:
      _updatePreviewFontPreset(payload.preset)
      break
    case EditorConstants.ActionTypes.CLEAR_PREVIEW_FONT:
      _clearPreviewFont()
      break
    default:
      break
  }
})
// No longer update fonts from style!
// when EditorConstants.ActionTypes.APPLY_STYLES # when hover or select a style
//   _updateFontsFromStyle(payload.styles)

if (!__SERVER_RENDERING__) {
  if (!window.DEBUG) {
    window.DEBUG = {}
  }
  window.DEBUG.FontStore = FontStore
}
export default FontStore
