// SPEC
// Owner: Jerry
// Feature name: HtmlComponent
// * Hover over the component, expect the overlay to show.
// * Click on the component, expect the App Store popup to open.
// * After interaction with App Store, clicking 'Save' should cause popup to close and iframe to refresh/resize.
// * This should also fire an API call to /s/components to save the component data.
// * Remove the section containing the component, expect an API call to /s/components to destroy the component data.
// * On editor init, expect the iframe to resize itself to a proper height.

import React from 'react'
import PropTypes from 'prop-types'
import ConfStore from 'js/stores/conf_store'

function eachSeries(items, timeoutInMilliseconds, fn) {
  if (items.length > 0) {
    let nextCalled = false
    const next = () => {
      if (nextCalled) {
        console.error('done function called after timeout')
        return
      }
      clearTimeout(timeout)
      nextCalled = true
      eachSeries(items.slice(1), timeoutInMilliseconds, fn)
    }
    const timeout = setTimeout(next, timeoutInMilliseconds)
    fn(items[0], next)
  }
}

let HtmlComponent = null
if (__NATIVE_WEB__) {
  const MobileDisabledNotice = require('js/components/MobileDisabledNotice')
  HtmlComponent = () => (
    <MobileDisabledNotice
      disabledNotice={__(
        'Mobile|App store is not yet editable on the mobile app.',
      )}
    />
  )
} else {
  const ReactDOM = require('react-dom')
  const _ = require('lodash')
  const $ = require('jquery')
  const ComponentFactory = require('js/utils/comp_factory')

  const PageMetaStore = require('../stores/page_meta_store')
  const EditorActions = require('../actions/editor_actions')
  const ComponentDataUtils = require('../utils/apis/components_api_utils')
  const MetaMixin = require('js/utils/mixins/meta_mixin')
  const CustomPropTypes = require('../utils/custom_prop_types')
  const AppStoreDialog = require('js/v3_bridge/app_store_dialog')
  const IframeHelper = require('../utils/helpers/IframeHelper')
  const logger = require('js/utils/logger')
  const loadFancyBox = import('js/vendor/jquery/jquery.fancybox3.js')

  const rt = require('./templates/html_component')

  const _htmlFieldsToSave = [
    'id',
    'value',
    'htmlValue',
    'selected_app_name',
    'page_id',
    'render_as_iframe',
    'app_list',
  ]
  const _bobcatPropTypes = {
    data: {
      id: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
      value: CustomPropTypes.html,
      render_as_iframe: PropTypes.bool,
      selected_app_name: PropTypes.string,
      app_list: PropTypes.string,
      binding: PropTypes.object,
    },
  }

  const _bobcatDefaultProps = () => ({
    data: {
      render_as_iframe: false,
      app_list: '{}',
    },
  })

  HtmlComponent = ComponentFactory.createPageComponent({
    displayName: 'HtmlComponent',
    mixins: [MetaMixin('editor')],

    bobcatPropTypes: _bobcatPropTypes,
    getBobcatDefaultProps: _bobcatDefaultProps,

    componentWillMount() {
      this.initMeta({
        iframeSrcQ: 0,
        canceled: false,
      })

      if (__IN_EDITOR__ && !this._hasId(this.props.id)) {
        return this._getId()
      }

      // if (!__IN_EDITOR__ && this._isTypeForm() && !__SERVER_RENDERING__) {
      //   this._applyTypeformScrollTopHack()
      // }
    },

    componentDidMount() {
      IframeHelper.startTimer()
      this._injectHtml()
      this._resizeIFrame()

      if (!__IN_EDITOR__ && this._isTypeForm()) {
        loadFancyBox.then(() => {
          this._initTypeForm()
        })
      }
    },

    componentDidUpdate(prevProps) {
      if (__IN_EDITOR__) {
        if (prevProps.value !== this.dtProps.value) {
          const dataToSave = _.pick(this.dtProps, _htmlFieldsToSave)
          this._saveComponent(dataToSave)
          this._injectHtml()
          return this._resizeIFrame()
        }
      }
    },

    componentWillUnmount() {
      if (this.props.isBlog && window.Ecwid) {
        window.Ecwid.destroy()
        window.Ecwid = null
      }
    },

    _initTypeForm() {
      const $this = $(ReactDOM.findDOMNode(this)).find('.type-form-popup')

      const basicClass = 'button-component'
      const styleClass = 'type-form-button-style'

      const $typeFormButton = $this.find('.type-form-button').eq(0)
      if ($.fn.fancybox) {
        const content = $typeFormButton.attr('data-iframe-content')
        const $virtualEl = $(content)
        const iframeSrc = $virtualEl.eq(0).attr('src')
        $typeFormButton.attr('data-src', iframeSrc)
        $typeFormButton.attr('data-href', 'javascript:;')
        $typeFormButton.fancybox({
          fullScreen: false,
          slideClass: 's-fancybox-typeform',
          iframe: {
            scrolling: 'auto',
          },
        })
      }
      const originalBg = $this.find('.type-form-button').css('background')
      $this.find('.type-form-button').removeClass(basicClass)
      const newBg = $this.find('.type-form-button').css('background')
      if (originalBg === newBg) {
        $this.addClass(styleClass)
      }
      $this.find('.type-form-button').addClass(basicClass)
    },

    _applyTypeformScrollTopHack() {
      // HACK: embed typeform iframe will scroll parent window to iframe position
      // but we can not prevent the code execution or the form will not show
      // so we manually check big jump
      let lastTop = $(window).scrollTop()
      let scrollCheck = null
      if (!$B.TH.isMobile()) {
        scrollCheck = function() {
          const currentTop = $(window).scrollTop()
          if (Math.abs(currentTop - lastTop) > 200) {
            $(window).scrollTop(lastTop)
          } else {
            lastTop = currentTop
          }
        }

        $(window).on('scroll', scrollCheck)

        setTimeout(() => {
          $(window).off('scroll', scrollCheck)
        }, 15000)
      }
    },

    _isTypeForm() {
      return this.props.selected_app_name === 'TypeFormApp'
    },

    _hasId(id) {
      return typeof id === 'number'
    },

    _getId() {
      this._setCanceled(false)
      return EditorActions[
        `createComponent${this.props.isBlog ? 'InBlog' : ''}`
      ]({
        data: {
          component: {},
        },
        success: data => {
          this.updateData({
            id: data.data.component.id,
          })
          return this.savePage()
        },
        error: data => {
          if (
            window.confirm(
              __(
                "Uh oh! There's been an error creating this HTML component. Try again?",
              ),
            )
          ) {
            return this._getId()
          }
          return this._setCanceled(true)
        },
      })
    },

    _resizeIFrame() {
      const $iframes = $(ReactDOM.findDOMNode(this)).find('iframe')
      if ($iframes.length) {
        return IframeHelper.resizeIFrames($iframes)
      }
    },

    _injectHtml() {
      if (!this.dtProps.render_as_iframe && !__IN_EDITOR__) {
        // HACK: use try catch to avoid breaking the entire rendering
        try {
          const htmlInjectDiv = ReactDOM.findDOMNode(this.refs.htmlInject)

          htmlInjectDiv.innerHTML = this._rawHtml()
          // We could use jQuery html to execute the script tags on insertion,
          // but we want to maintain their execution order and wait for each
          // script to load one at a time
          eachSeries(
            $(htmlInjectDiv).find('script'),
            4000, // need timeout because don't get a failure result from runScript
            (script, done) => {
              const rscriptType = /^$|\/(?:java|ecma)script/i // taken from jQuery .html source
              if (script.src && rscriptType.test(script.type || '')) {
                $.getScript(script.src).done(done)
              } else {
                const rcleanScript = /^\s*<!(?:\[CDATA\[|--)|(?:\]\]|--)>\s*$/g // from jQuery
                $.globalEval(script.textContent.replace(rcleanScript, ''))
                return done()
              }
            },
          )
        } catch (error) {
          // use console.log because user might want to see this error
          logger.log(`Html section script error: ${error}`)
          return $(ReactDOM.findDOMNode(this.refs.htmlInject)).append(
            `Script error: ${error}`,
          )
        }
      }
    },

    _hasContent() {
      return this.dtProps.value.length > 0
    },

    _renderAsIframe() {
      return this.dtProps.render_as_iframe
    },

    _rawHtml() {
      return _.unescape(this.dtProps.value || '')
    },

    _onClickEditor() {
      const cb = data => {
        if (data.id === this.dtProps.id) {
          const dataToSave = _.pick(data, _htmlFieldsToSave)
          this.updateData(dataToSave)
          return this.savePage()
        }
        return window.error(
          __(
            "Uh oh! There's been an error saving this HTML component. Try again.",
          ),
        )
      }
      if (this.props.isBlog) {
        var dialog = new AppStoreDialog(
          _.extend({}, this.dtProps, {
            htmlValue: this._rawHtml(),
            page_id: PageMetaStore.getId(),
          }),
          data => {
            cb(data)
            return dialog.close()
          },
          () => dialog.close(),
        )
        // let dialog = new AppStoreDialog(
        //   _.extend({}, this.dtProps, {
        //     htmlValue: this._rawHtml(),
        //     page_id: PageMetaStore.getId(),
        //   }),
        //   (data) => {
        //     cb(data)
        //     dialog.close()
        //   }, () => {
        //     dialog.close()
        //   }
        // )
      } else {
        return EditorActions.openAppStoreDialog(
          _.extend({}, this.dtProps, {
            htmlValue: this._rawHtml(),
            page_id: PageMetaStore.getId(),
          }),
          cb,
        )
      }
    },

    _saveComponent(data) {
      if (this.props.isBlog) {
        ComponentDataUtils.update(this.dtProps.id, data, this._reloadIframe)
      } else {
        // TODO: Using promise instead of callback
        return EditorActions.saveHTMLComponent(
          this.dtProps.id,
          data,
          this._reloadIframe,
        )
      }
    },
    // ComponentDataUtils.update(@dtProps.id, data, @_reloadIframe)

    _iframeSrcQ() {
      return this.getMeta('iframeSrcQ')
    },

    _reloadIframe() {
      return this.updateMeta({ iframeSrcQ: this.getMeta('iframeSrcQ') + 1 })
    },

    // for admin use only to fix component that have repeated id
    _recreateComponent() {
      if (
        window.confirm(
          'Recreating will delete any existing component! Make sure you understand what this does',
        )
      ) {
        this.updateData(_bobcatDefaultProps().data)
        return this._getId()
      }
    },

    render() {
      if (this._getCanceled()) {
        return (
          <div
            className="s-common-status"
            style={{ cursor: 'pointer' }}
            onClick={this._getId}>
            {__('Click here to create HTML component again.')}
          </div>
        )
      } else if (!this._hasId(this.props.id)) {
        return __IN_EDITOR__ ? (
          <div className="s-loading-wrapper">
            <div className="s-loading" />
          </div>
        ) : null
      }
      return rt.apply(this)
    },
  })
}

export default HtmlComponent
