require('./init')

import $ from 'jquery'
import 'js/vendor/jquery/easing'
import React from 'react'
import ReactDOM from 'react-dom'
import { Iterable } from 'immutable'
import { tct } from 'r-i18n'

import { createStore, applyMiddleware } from 'redux'
import { createLogger } from 'redux-logger'
import { Provider } from 'react-redux'
import thunkMiddleware from 'redux-thunk'
import ManagerReducer from 'nextgen/ecommerce/manager/reducers/ManagerReducer'
import connectReduxStoreToBinding from 'js/utils/helpers/connectReduxStoreToBinding'

import EcommerceManagerStore from 'js/stores/EcommerceManagerStore'
import FeatureStore from 'js/stores/FeatureStore'
import ConfStore from 'js/stores/conf_store'
import 'js/reactInit.es6'

import * as UrlConstants from 'js/constants/url_constants'

import ProductPanel from 'nextgen/ecommerce/manager/components/productPanel'
import CategoryManagerWrapper from 'nextgen/ecommerce/manager/components/settings/categoryManagerWrapper'
import MembershipManager from 'nextgen/ecommerce/manager/components/settings/membershipManager'
import PaymentGatewaySettingsPanel from 'nextgen/ecommerce/manager/components/settings/paymentChannel/PaymentGatewaySettingsPanel'
// import Orders from 'nextgen/manager/Products'
// import Coupons from 'nextgen/manager/Products'
// import Settings from 'nextgen/manager/Products'

import gonReader from 'js/utils/gonReader'
import { getSupportCurrencies } from 'js/utils/helpers/EcommerceHelper'
import * as RHL from 'react-hot-loader'
import wrapErrorBoundary from 'js/components/ErrorBoundary'

import { UPDATE_SETTINGS_SUCCESS } from 'nextgen/ecommerce/manager/actions/entities/settings'
import { initializeAccountFromV3 } from 'nextgen/ecommerce/manager/actions/entities/accounts'

import EcommerceManagerBridge from 'js/v4_bridge/EcommerceManagerBridge'

import * as i18nHelper from 'js/utils/helpers/i18nHelper'

import ComponentKitContext from 'js/utils/ComponentKitContext'

const AppContainer = wrapErrorBoundary(RHL.AppContainer)

const p1 = import(`locales/${i18nHelper.getTranslationFile()}`)

const loggerMiddleware = createLogger({
  collapsed: true,
  predicate: (getState, action) => process.env.NODE_ENV !== 'production',
  stateTransformer: state => {
    if (Iterable.isIterable(state)) {
      return state.toJS()
    }
    return state
  },
})

const store = createStore(
  ManagerReducer,
  applyMiddleware(
    thunkMiddleware, // lets us dispatch() functions
    loggerMiddleware, // neat middleware that logs actions
  ),
)

const baseProps = {
  siteId: gonReader('id'),
  siteState: gonReader('state'),
  publicUrl: gonReader('public_url'),
  meta: gonReader('pageOwner.ecommerce'),
  isStandAlone: window.parent === window,
  isPro: ['pro', 'namecheap', 'sxlbiz'].includes(
    gonReader('pageOwner.membership'),
  ),
  isVip: gonReader('pageOwner.membership') === 'vip',
  isSxl: Boolean(gonReader('globalConf.isSxl')),
  isAdmin: Boolean(gonReader('pageOwner.isAdmin')),
  isSupport: Boolean(gonReader('pageOwner.isSupport')),
  supportedCurrency: gonReader('conf.SUPPORTED_CURRENCY'),
}

const productProps = {
  ...baseProps,
  productLimit:
    gonReader('pageOwner.ecommerce').productLimit ||
    (gonReader('globalConf.isSxl') ? 5 : 1),
  upgradeUrl: UrlConstants.PRICING.GOTO_AND_RETURN(
    gonReader('id'),
    'ecommerce',
  ),
  productPageRollout: gonReader('globalConf.rollout.product_page'),
  productDetailRollout: gonReader('globalConf.rollout.product_detail'),
  siteMemberShip: gonReader('globalConf.rollout.siteMemberShip'),
  noCategoryLimit: gonReader('globalConf.rollout.no_category_limit'),
}

const supportedChannels = () => {
  const isSxl = Boolean(gonReader('globalConf.isSxl'))
  const paypal = {
    channel: 'paypal',
    supportedCurrencies: getSupportCurrencies('paypal'),
    features: [
      __('Ecommerce|~2.9% + 30¢ per transaction'),
      __('Ecommerce|Get funds in just a few minutes'),
      tct(__('Ecommerce|Accepts [link]'), {
        link: (
          <span>
            <i className="fa fa-cc-visa" aria-hidden="true" />
            <i className="fa fa-cc-mastercard" aria-hidden="true" />
            <i className="fa fa-cc-amex" aria-hidden="true" />
            <i className="fa fa-cc-discover" aria-hidden="true" />
          </span>
        ),
      }),
      __('Ecommerce|No transaction fees from Strikingly'),
    ],
    logo: '/images/ecommerce/paypal.png',
    methods: '',
    additionalInfo: tct(__('Ecommerce|[link: Learn more about PayPal.]'), {
      link: <a target="_blank" href="https://www.paypal.com/" />,
    }),
  }
  const stripe = {
    channel: 'stripe',
    supportedCurrencies: getSupportCurrencies('stripe'),
    features: [
      __('Ecommerce|~2.9% + 30¢ per transaction'),
      __('Ecommerce|Get funds in about 7 days'),
      tct(__('Ecommerce|Accepts [link]'), {
        link: (
          <span>
            <i className="fa fa-cc-visa" aria-hidden="true" />
            <i className="fa fa-cc-mastercard" aria-hidden="true" />
            <i className="fa fa-cc-amex" aria-hidden="true" />
            <i className="fa fa-cc-discover" aria-hidden="true" />
            <i className="fa fa-cc-jcb" aria-hidden="true" />
          </span>
        ),
      }),
      __('Ecommerce|No transaction fees from Strikingly'),
    ],
    logo: '/images/ecommerce/stripe-logo.png',
    methods: '',
    additionalInfo: tct(__('Ecommerce|[link: Learn more about Stripe.]'), {
      link: <a target="_blank" href="https://stripe.com/" />,
    }),
  }
  const alipay = {
    channel: 'alipay',
    supportedCurrencies: getSupportCurrencies('alipay'),
    features: [
      __('Ecommerce|Needs business bank account and ICP license'),
      __('Ecommerce|Get funds immediately after payment'),
      __('Ecommerce|0.6% per successful charge from Alipay'),
      __('Ecommerce|No transaction fees from Strikingly'),
    ],
    logo: '/images/ecommerce/ic_alipay.png',
    methods: __('Ecommerce|Instant Payment'),
    additionalInfo: tct(
      __(
        'Ecommerce|Enter your PID and MD5 Key to enable your Alipay account.[link: Learn more about how to get PID and MD5 Key from Alipay manage platform.]',
      ),
      {
        link: (
          <a
            target="_blank"
            href="http://help.sxl.cn/hc/zh-cn/articles/115000046301"
          />
        ),
      },
    ),
  }
  const wechatpay = {
    channel: 'wechatpay',
    supportedCurrencies: getSupportCurrencies('wechatpay'),
    features: [
      __('Ecommerce|Needs business bank account and ICP license'),
      __('Ecommerce|Get funds immediately after payment'),
      __('Ecommerce|0.6% per successful charge from WeChat Pay'),
      __('Ecommerce|No transaction fees from Strikingly'),
    ],
    logo: '/images/ecommerce/ic_wechat_pay.png',
    methods: __('Ecommerce|In-App Web-based Payment'),
    additionalInfo: tct(
      __(
        'Ecommerce|Enter your WeChat Pay account info below to enable your Alipay account.[link: Learn more about how to get WeChat Pay account info.]',
      ),
      {
        link: (
          <a
            target="_blank"
            href="http://help.sxl.cn/hc/zh-cn/articles/115000046661"
          />
        ),
      },
    ),
  }
  const offline = {
    channel: 'offline',
    supportedCurrencies: getSupportCurrencies('offline'),
    features: [],
    logo: '',
    methods: '',
    additionalInfo: tct(__('Ecommerce|[link: Learn more]'), {
      link: (
        <a
          target="_blank"
          href={
            isSxl
              ? 'http://help.sxl.cn/hc/zh-cn/articles/115000100782'
              : 'https://support.strikingly.com/hc/en-us/articles/115000100802'
          }
        />
      ),
    }),
    description: __(
      'Ecommerce|Customers will skip payment while placing the order. You must set up offline payment yourself!',
    ),
  }

  const midtrans = {
    channel: 'midtrans',
    supportedCurrencies: getSupportCurrencies('midtrans'),
    features: [
      __(
        'Ecommerce|Accept card payment, bank transfer, direct debit, e-wallet, over the counter',
      ),
      __('Ecommerce|Transaction fee varies with different payment methods'),
      __('Ecommerce|No transaction fees from Strikingly'),
    ],
    logo: '/images/ecommerce/midtrans.png',
    methods: '',
  }

  let supportedChannelsConf = []
  if (isSxl) {
    supportedChannelsConf = [paypal, alipay, wechatpay]
  } else {
    supportedChannelsConf = [paypal, stripe]
  }

  // Rollout for Midtrans payments
  if (!isSxl && ConfStore.getMidtransPayments()) {
    supportedChannelsConf.push(midtrans)
  }

  // Rollout for offline payments
  if (ConfStore.getOfflinePayments()) {
    supportedChannelsConf.push(offline)
  }

  return supportedChannelsConf
}

const KitWrappedProductPanel = ComponentKitContext(ProductPanel)

class ProductsContainer extends React.Component {
  componentWillMount() {
    EcommerceManagerBridge.subPublishUrlChange((topic, data) => {
      baseProps.publicUrl = data.url
      this.forceUpdate()
    })

    EcommerceManagerBridge.subSiteStateChange((topic, data) => {
      baseProps.siteState = data.state
      this.forceUpdate()
    })
  }

  render() {
    return (
      <Provider store={store}>
        <KitWrappedProductPanel
          {...productProps}
          currentProductDetail={this.props.currentProductDetail}
          canUseCategory={FeatureStore.canUse('ecommerce_category')}
          canSeeCategory={FeatureStore.canSee('ecommerce_category')}
        />
      </Provider>
    )
  }
}

class CategoryContainer extends React.Component {
  render() {
    return (
      <Provider store={store}>
        <CategoryManagerWrapper
          {...productProps}
          canUseCategory={FeatureStore.canUse('ecommerce_category')}
          canSeeCategory={FeatureStore.canSee('ecommerce_category')}
        />
      </Provider>
    )
  }
}

const CategoryContainerWithComponentKit = ComponentKitContext(CategoryContainer)

class MembershipContainer extends React.Component {
  render() {
    return (
      <Provider store={store}>
        <MembershipManager {...productProps} />
      </Provider>
    )
  }
}

const PaymentSettingsContainer = () => (
  <Provider store={store}>
    <PaymentGatewaySettingsPanel
      {...baseProps}
      supportedChannels={supportedChannels()}
    />
  </Provider>
)

EcommerceManagerBridge.subSettingsChange((topic, data) => {
  const siteId = $S.id.toString()
  let settings =
    store
      .getState()
      .getIn(['entities', 'settings', 'data', siteId, 'settings']) || {}
  if (settings.toJS) {
    settings = settings.toJS()
  }

  store.dispatch({
    type: UPDATE_SETTINGS_SUCCESS,
    payload: {
      settings: Object.assign(settings, data.settings),
    },
    meta: {
      siteId,
    },
  })
  store.dispatch(initializeAccountFromV3(settings.accounts))
})

const siteIdString = $S.id.toString()
const prevStates = {
  product: store
    .getState()
    .getIn(['entities', 'product', 'data', siteIdString]),
  settings: store
    .getState()
    .getIn(['entities', 'settings', 'data', siteIdString, 'settings']),
  category: store.getState().getIn(['entities', 'category', 'data']),
}
store.subscribe(a => {
  const newProductsState = store
    .getState()
    .getIn(['entities', 'product', 'data', $S.id.toString()])
  const newSettingsState = store
    .getState()
    .getIn(['entities', 'settings', 'data', siteIdString, 'settings'])
  const newCategoryState = store
    .getState()
    .getIn(['entities', 'category', 'data'])

  if (newProductsState !== prevStates.product) {
    prevStates.product = newProductsState
    EcommerceManagerBridge.pubProductsChange(_.flatten(newProductsState.toJS()))
  }

  if (newSettingsState !== prevStates.settings) {
    prevStates.settings = newSettingsState
    EcommerceManagerBridge.pubSettingsChange(newSettingsState.toJS())
  }

  if (newCategoryState !== prevStates.category) {
    prevStates.category = newCategoryState
    EcommerceManagerBridge.pubCategoriesChange(
      _.flatten(newCategoryState.toJS()),
    )
  }
})

Promise.all([p1])
  .then(([poFile]) => {
    const I18n = require('js/utils/i18n')
    I18n.init(poFile)
    $(() => {
      const cloudinary = require('cloudinary')
      cloudinary.config('cloud_name', ConfStore.getCloudinaryCloudName())

      if (window.parent === window) {
        let ImageAssetDialog = require('js/v4_bridge/react_app_bridge/ImageAssetDialogApp')
        const EditorStore = require('js/stores/editor_store')
        const ctx = EditorStore.init()
        connectReduxStoreToBinding(store, ctx.getBinding())
        ImageAssetDialog = ctx.bootstrap(ImageAssetDialog())
        ReactDOM.render(
          <Provider store={store}>
            <AppContainer>
              <ImageAssetDialog />
            </AppContainer>
          </Provider>,
          document.getElementById('container'),
        )
      }

      EcommerceManagerStore.init()
      FeatureStore.hydrate($S.features)

      connectReduxStoreToBinding(store, EcommerceManagerStore.getBinding())

      // render counpon part for ecommerce manager
      const couponsCtx = EcommerceManagerStore.getCouponsContext()
      const CouponManager = require('js/components/ecommerce/manager/coupon/CouponManager')
      const WrappedCoupon = ComponentKitContext(
        couponsCtx.bootstrap(CouponManager),
      )
      ReactDOM.render(
        <Provider store={store}>
          <AppContainer>
            <WrappedCoupon />
          </AppContainer>
        </Provider>,
        document.getElementById('coupon-container'),
      )

      /*
      Add morearty binding to product container
      currentProductDetail is a data container and binding path never get changed, only data can be set
    */
      const productsContext = EcommerceManagerStore.getProductsContext()
      const currentProductDetail = EcommerceManagerStore.getProductsBinding().sub(
        'currentProductDetail',
      )
      const WrapperedProductsContainer = productsContext.bootstrap(
        ProductsContainer,
      )
      ReactDOM.render(
        <AppContainer>
          <WrapperedProductsContainer
            currentProductDetail={currentProductDetail}
          />
        </AppContainer>,
        document.getElementById('products-container'),
      )
      ReactDOM.render(
        <AppContainer>
          <PaymentSettingsContainer />
        </AppContainer>,
        document.getElementById('payment-manager'),
      )
      ReactDOM.render(
        <CategoryContainerWithComponentKit />,
        document.getElementById('category-manager'),
      )
      ReactDOM.render(
        <AppContainer>
          <MembershipContainer />
        </AppContainer>,
        document.getElementById('membership-manager'),
      )
    })
  })
  .catch(e => console.error(e))

if (module.hot) {
  module.hot.accept(err => {
    if (err) {
      console.error(err)
    }
  })
}
