import _ from 'lodash'

import PageDataStore from 'js/stores/page_data_store'
import DeviceHelper from 'js/utils/helpers/device_helper'
import { getFormattedPrice } from 'js/utils/EcommerceUtils'
import { traverseObj } from 'common/utils/lodashb'
import * as Immutable from 'immutable'
import EcommerceConstants from 'js/constants/ecommerce_constants'
import { flatten, path, pathEq } from 'ramda'

const { ZERO_DECIMAL_CURRENCY_LIST } = EcommerceConstants

function _getSettings() {
  const EcommerceStore = require('js/stores/ecommerce_store')
  return EcommerceStore.getSettings()
}

function _getCart() {
  const EcommerceBuyStore = require('js/stores/EcommerceBuyStore')
  return EcommerceBuyStore.getCart()
}

function _getCurrentDevice() {
  if (!DeviceHelper.isWechat() && DeviceHelper.isMobile()) {
    return 'mobile'
  }
  if (DeviceHelper.isWechat()) {
    return 'wechat'
  }
  return 'desktop'
}

function _getSuportedChannelsWithCurrentDevice(channels) {
  const __hasProp = {}.hasOwnProperty
  const deviceedChannels = {
    wechat: [],
    mobile: [],
    desktop: [],
  }

  for (const key in channels) {
    if (!__hasProp.call(channels, key)) {
      continue
    }
    const value = channels[key]
    if (value) {
      switch (key) {
        case 'stripe':
        case 'alipay':
          deviceedChannels.mobile.push(key)
          deviceedChannels.desktop.push(key)
          break
        case 'paypal':
        case 'offline':
          deviceedChannels.mobile.push(key)
          deviceedChannels.wechat.push(key)
          deviceedChannels.desktop.push(key)
          break
        case 'wechatpay':
          deviceedChannels.wechat.push(key)
          deviceedChannels.desktop.push(key)
          break
        case 'pingppAlipayWap':
          deviceedChannels.mobile.push(key)
          break
        case 'pingppAlipayQr':
        case 'pingppWxPubQr':
          deviceedChannels.desktop.push(key)
          break
        case 'pingppWxPub':
          deviceedChannels.wechat.push(key)
          break
        default:
          break
      }
    }
  }
  return deviceedChannels
}

// function _getProducts() {
//   return EcommerceStore.getProducts()
// }

function _addCurrencySymbol(price) {
  const settings = _getSettings()
  return getFormattedPrice(price, settings.currencyData)
}

function _getCountryByCode(countryCode) {
  return ($S.country_list || {})[countryCode]
}

function _removeHttpProtocol(url) {
  if (!url) {
    return ''
  }
  return url.replace('http:', '')
}

export default {
  availableDevicesToPayment() {
    const settings = _getSettings()
    const deviceedChannels = _getSuportedChannelsWithCurrentDevice(
      settings.paymentGateways,
    )
    const supportedDevices = []
    if (deviceedChannels.wechat.length > 0) {
      supportedDevices.push('wechat')
    }
    if (deviceedChannels.mobile.length > 0) {
      supportedDevices.push('mobile')
    }
    if (deviceedChannels.desktop.length > 0) {
      supportedDevices.push('desktop')
    }
    return supportedDevices
  },

  hasAvailablePaymentWithCurrentDevice() {
    const channels = this.getAvailableChannelsWithCurrentDevice()
    return channels.length > 0
  },

  getAvailableChannelsWithCurrentDevice() {
    const settings = _getSettings()
    let channels = []

    if (PageDataStore.hasSection('ecommerce')) {
      const device = _getCurrentDevice()
      const deviceedChannels = _getSuportedChannelsWithCurrentDevice(
        settings.paymentGateways,
      )
      channels = deviceedChannels[device]
    }
    return channels
  },

  getAvailableChannels(
    channels = [
      'paypal',
      'stripe',
      'alipay',
      'pingppAlipayQr',
      'pingppAlipayWap',
      'pingppWxPub',
      'pingppWxPubQr',
      'wechatpay',
      'offline',
      'midtrans',
    ],
  ) {
    const settings = _getSettings()
    const availableChannels = []
    const { currencyCode, paymentGateways } = settings
    const self = this
    channels.map(channel => {
      const supportCurrencies = self.getSupportCurrencies(channel)
      if (
        supportCurrencies.includes(currencyCode.toString()) &&
        paymentGateways[channel]
      ) {
        availableChannels.push(channel)
      }
    })
    return availableChannels
  },

  getAvailableChannelsCount(
    channels = [
      'paypal',
      'stripe',
      'alipay',
      'pingppAlipayQr',
      'pingppAlipayWap',
      'pingppWxPub',
      'pingppWxPubQr',
      'wechatpay',
      'offline',
      'midtrans',
    ],
  ) {
    return this.getAvailableChannels(channels).length
  },

  isPaymentAccountSet() {
    if (PageDataStore.hasSection('ecommerce')) {
      const paymentGatewaysCount = this.getAvailableChannelsCount()
      return paymentGatewaysCount > 0
    }
    return true
  },

  pageHasCoupon() {
    return _getSettings().hasCoupon
  },

  userHasCoupon(coupon = _getCart().coupon) {
    if (coupon && coupon.category) {
      return true
    }
  },

  userHasCouponWithType(type, coupon = _getCart().coupon) {
    if (this.userHasCoupon(coupon) && coupon.category === type) {
      return true
    }
  },

  isInCondition(condition, coupon = _getCart().coupon) {
    if (this.userHasCoupon() && coupon.option.condition[condition]) {
      return true
    }
  },

  needToShowDiscountInfo(coupon = _getCart().coupon) {
    return this.userHasCoupon()
  },

  addCurrencySymbol: _addCurrencySymbol,

  getPriceScope(product) {
    const priceList =
      product && product.variations
        ? product.variations.map(variation => variation.price)
        : [0]
    const minPrice = Math.min(...priceList)
    const maxPrice = Math.max(...priceList)

    let rs
    if (minPrice === maxPrice) {
      rs = _addCurrencySymbol(minPrice)
    } else {
      rs = `${_addCurrencySymbol(minPrice)} - ${_addCurrencySymbol(maxPrice)}`
    }

    return rs
  },

  getDecimalNum(currency = _getSettings().currencyCode) {
    let rs = 2
    if (ZERO_DECIMAL_CURRENCY_LIST.indexOf(currency) !== -1) {
      rs = 0
    }
    return rs
  },

  getDiscountItem() {
    if (!this.userHasCoupon()) {
      return null
    }

    const { items: itemsInCart, coupon } = _getCart()
    const amount = coupon.option.amount
    const discountProductId = coupon.option.condition.productId

    const itemsInProduct = itemsInCart.filter(
      item => String(item.productId) === String(discountProductId),
    )

    if (itemsInProduct.length) {
      const maxPriceItem = itemsInProduct.reduce((acc, item) => {
        if (
          acc &&
          Number(acc.orderItem.price) >= Number(item.orderItem.price)
        ) {
          return acc
        } else {
          return item
        }
      }, null)

      return maxPriceItem
    } else {
      return null
    }
  },

  getDiscountForProductCondition() {
    if (!this.userHasCoupon()) {
      return 0
    }

    const discountItem = this.getDiscountItem()

    if (discountItem) {
      const { coupon } = _getCart()
      const amount = coupon.option.amount

      return (discountItem.orderItem.price * amount) / 100
    } else {
      return 0
    }
  },

  getDiscountNum() {
    const { coupon } = _getCart()
    let number = 0
    if (!this.needToShowDiscountInfo()) {
      return number
    }

    if (this.userHasCoupon()) {
      switch (coupon.category) {
        case 'free_shipping':
          number = this.getShippingFeeNum()
          return number
        case 'flat':
          number = coupon.option.amount / 100
          break
        case 'percentage':
          number = this.isInCondition('productId')
            ? this.getDiscountForProductCondition()
            : (this.getTotalItemPriceNum() * coupon.option.amount) / 100

          if (this.getDecimalNum() === 0) {
            number = Math.round(number)
          }
          break
        // no default
      }
    }

    if (number >= this.getTotalItemPriceNum()) {
      number = this.getTotalItemPriceNum()
    }

    return number
  },

  getShippingFeeNum() {
    const settings = _getSettings()
    const { orderData, shipping } = _getCart()
    let shippingFee = 0

    if (!orderData || !shipping || !settings.shippingRegions) {
      return 0
    }

    if (!orderData.shipping && !Array.isArray(settings.shippingRegions)) {
      const countryCode = shipping.country ? shipping.country.value : false
      if (!countryCode) {
        return 0
      }
      const feeData =
        settings.shippingRegions[countryCode] ||
        settings.shippingRegions[
          _getCountryByCode(countryCode)
            ? _getCountryByCode(countryCode).continent
            : undefined
        ] ||
        settings.shippingRegions.default

      if (!feeData) {
        return 0
      }

      const decimalNum = this.getDecimalNum()
      const feePerAdditionalItem = (
        feeData.feePerAdditionalItem / 100.0
      ).toFixed(decimalNum)
      const feePerOrder = (feeData.feePerOrder / 100.0).toFixed(decimalNum)

      const { items } = _getCart()
      if (_.all(items, i => !i.product.shippingInfo)) {
        return 0
      }

      const additionalFee = items.reduce((total, next) => {
        const fee = next.product.shippingInfo
          ? feePerAdditionalItem * next.orderItem.quantity
          : 0
        return total + fee
      }, 0)

      shippingFee = feePerOrder - feePerAdditionalItem + additionalFee
    } else {
      shippingFee = orderData.shipping / 100.0
    }

    return shippingFee
  },

  getTotalItemPriceNum() {
    const { items, coupon } = _getCart()

    let totalPrice = _.reduce(
      items,
      (total, next) => total + next.orderItem.price * next.orderItem.quantity,
      0,
    )

    if (this.getDecimalNum() === 0) {
      totalPrice = Math.round(totalPrice)
    }

    return totalPrice
  },

  getTotalNum() {
    let amount = 0
    if (!this.userHasCoupon()) {
      return (
        this.getShippingFeeNum() +
        this.getTotalItemPriceNum() +
        this.getTaxesNum()
      )
    }

    if (this.userHasCouponWithType('free_shipping')) {
      return this.getTotalItemPriceNum() + this.getTaxesNum()
    }

    amount = this.getTotalItemPriceNum() - this.getDiscountNum()
    if (amount < 0) {
      amount = 0
    }

    return amount + this.getShippingFeeNum() + this.getTaxesNum()
  },

  getTaxesNum() {
    const settings = _getSettings()
    const taxesRate = (settings.taxes || 0) / 100
    const itemPrice = this.getTotalItemPriceNum()
    const discountForTax = this.userHasCouponWithType('free_shipping')
      ? 0
      : this.getDiscountNum()
    let taxesNum = 0

    if (!taxesRate) {
      return taxesNum
    }

    taxesNum = ((itemPrice - discountForTax) * taxesRate) / (1 + taxesRate)
    if (this.getDecimalNum() === 0) {
      taxesNum = Math.round(taxesNum)
    }

    return taxesNum
  },

  getDiscountDescription(coupon = _getCart().coupon, options) {
    let description = ''
    if (this.userHasCouponWithType('free_shipping', coupon)) {
      description = __('EcommerceCoupon|Free Shipping')
    } else if (this.userHasCouponWithType('percentage', coupon)) {
      if (this.isInCondition('productId')) {
        const item = this.getDiscountItem()
        description = __(
          'EcommerceCoupon|%{amount} off for %{productCount} %{productName} ',
          {
            productCount: options ? options.productCount : 1,
            productName: options ? options.productName : item.product.name,
            amount: `${coupon.option.amount}%`,
          },
        )
      } else {
        description = __('EcommerceCoupon|%{amount} Off', {
          amount: `${coupon.option.amount}%`,
        })
      }
    } else if (this.userHasCouponWithType('flat', coupon)) {
      const decimalNum = this.getDecimalNum()
      const fee = (coupon.option.amount / 100).toFixed(decimalNum)
      description = __('EcommerceCoupon|%{amount} Off', {
        amount: this.addCurrencySymbol(fee),
      })
    }
    return description
  },

  getDiscount() {
    return `- ${this.addCurrencySymbol(this.getDiscountNum())}`
  },

  getShippingFee() {
    return this.addCurrencySymbol(this.getShippingFeeNum())
  },

  getSubtotal() {
    return this.addCurrencySymbol(this.getTotalItemPriceNum())
  },

  getTaxes() {
    return this.addCurrencySymbol(this.getTaxesNum())
  },

  getSubtotalWithDiscount() {
    if (this.userHasCouponWithType('free_shipping')) {
      return this.addCurrencySymbol(this.getTotalItemPriceNum())
    }
    return this.addCurrencySymbol(
      this.getTotalItemPriceNum() - this.getDiscountNum(),
    )
  },

  getTotalPrice() {
    return this.addCurrencySymbol(this.getTotalNum())
  },

  getTrackData() {
    if (__PRODUCT_NAME__ === 'sxl') {
      return {}
    }

    const cartData = _getCart()
    const trackData = {
      currency: _getSettings().currencyData.code,
      content_type: 'product',
      content_ids: [],
      content_name: [],
      value: this.getTotalNum(),
      num_items: 0,
    }
    cartData.items.forEach(item => {
      const { orderItem } = item
      trackData.content_name.push((item.product && item.product.name) || '')
      trackData.content_ids.push(item.product.id)
      trackData.num_items += Number(orderItem.quantity)
    })

    return trackData
  },

  loadDistrictsAsync(selectedCode = '000000', callback = () => {}) {
    if (__PRODUCT_NAME__ === 'sxl') {
      require.ensure([], require => {
        const asyncResult = import(`../../data/china_districts/${selectedCode}.json`)
        asyncResult().then(json => callback(json))
      })
    }
  },

  /**
   * @param {conf} {string} example as following
   *  {
   *    "appId" ： "wx2421b1c4370ec43b",     //公众号名称，由商户传入
   *    "timeStamp"：" 1395712654",         //时间戳，自1970年以来的秒数
   *    "nonceStr" ： "e61463f8efa94090b1f366cccfbbb444", //随机串
   *    "package" ： "prepay_id=u802345jgfjsdfgsdg888"
   *    "signType" ： "MD5",         //微信签名方式
   *    "paySign" ： "70EA570631E4BB79628FBCA90534C63FF7FADD89" //微信签名
       }
   */
  useWechatpay(conf, cb) {
    if (DeviceHelper.isWechat()) {
      if (typeof WeixinJSBridge === 'undefined') {
        alert('请确认您的网站已正确加载微信sdk')
        return false
      }
      WeixinJSBridge.invoke('getBrandWCPayRequest', conf, res => {
        if (res.err_msg === 'get_brand_wcpay_request:ok') {
          // 使用以上方式判断前端返回,微信团队郑重提示：res.err_msg, 将在用户支付成功后返回ok，但并不保证它绝对可靠
          cb && cb()
        }
      })
    } else {
      // pay on desktop with wechat qr code scanning
      const EcommerceActions = require('js/actions/EcommerceActions')
      EcommerceActions.gotoEcommerceBuyDialog('paymentqr', true)
    }
  },
  getInitialPaymentAccount(provider) {
    switch (provider) {
      case 'alipay':
        return { pid: null, md5Key: null }
      case 'wechatpay':
        return { mchId: null, appId: null, apiKey: null, appSecret: null }
      case 'paypal':
        return { email: null }
      case 'offline':
        return { instructions: null, method: null }
      case 'midtrans':
        return { merchantId: null, clientKey: null, serverKey: null }
      default:
        return {}
    }
  },

  getSupportCurrencies(provider) {
    switch (provider) {
      case 'alipay':
      case 'wechatpay':
      case 'pingppAlipayQr':
      case 'pingppAlipayWap':
      case 'pingppWxPub':
      case 'pingppWxPubQr':
        return ['CNY']
      case 'paypal':
        return [
          'AUD',
          'BRL',
          'CAD',
          'CHF',
          'CZK',
          'DKK',
          'EUR',
          'GBP',
          'HKD',
          'HUF',
          'ILS',
          'JPY',
          'KRW',
          'MXN',
          'MYR',
          'NOK',
          'NZD',
          'PHP',
          'PLN',
          'RUB',
          'SEK',
          'SGD',
          'THB',
          'TWD',
          'USD',
        ]
      case 'stripe':
        return [
          'AUD',
          'BRL',
          'CAD',
          'CHF',
          'CLP',
          'DKK',
          'EUR',
          'GBP',
          'HKD',
          'IDR',
          'INR',
          'JPY',
          'KRW',
          'MXN',
          'MYR',
          'NOK',
          'NZD',
          'PHP',
          'SEK',
          'SGD',
          'THB',
          'TWD',
          'USD',
          'COP',
          'VND',
          'ZAR',
        ]
      case 'offline':
        return [
          'AUD',
          'BDT',
          'BRL',
          'CAD',
          'CHF',
          'CLP',
          'CNY',
          'CZK',
          'DKK',
          'EUR',
          'GBP',
          'HKD',
          'HUF',
          'IDR',
          'ILS',
          'INR',
          'JPY',
          'KRW',
          'MXN',
          'MYR',
          'NOK',
          'NZD',
          'PEN',
          'PHP',
          'PLN',
          'RUB',
          'SEK',
          'SGD',
          'THB',
          'TWD',
          'USD',
          'COP',
          'VND',
          'ZAR',
        ]
      case 'midtrans':
        return ['IDR']
      default:
        return []
    }
  },

  getProviderNameMap(provider) {
    switch (provider) {
      // TODO: Server has two name for stripe. On disconnect and accounts.
      case 'stripe':
        return 'stripe_connect'
      case 'stripe_connect':
        return 'stripe'
      default:
        return provider
    }
  },

  removeHttpProtocolForImageUrl(products) {
    traverseObj(products, obj => {
      if (obj.thumbnailUrl) {
        obj.thumbnailUrl = _removeHttpProtocol(obj.thumbnailUrl)
      }
      if (obj.url) {
        obj.url = _removeHttpProtocol(obj.url)
      }
    })

    return products
  },

  needNarrowCurrencySymbol() {
    const settings = _getSettings()
    return settings.currencyCode == 'KRW'
  },

  isEmptyDimension(dimension) {
    const dimen = Immutable.isImmutable(dimension)
      ? dimension.toJS()
      : dimension
    if (
      dimen === undefined ||
      dimen.name === '' ||
      dimen.options === undefined ||
      dimension.options === null ||
      dimen.options.length === 0
    ) {
      return true
    } else {
      return false
    }
  },

  hasMutipleDimensions(dimensions) {
    if (!dimensions) {
      return false
    }

    const { dimension1, dimension2 } = Immutable.isImmutable(dimensions)
      ? dimensions.toJS()
      : dimensions
    return (
      !this.isEmptyDimension(dimension1) && !this.isEmptyDimension(dimension2)
    )
  },
  normalizeItemName(name) {
    if (name && name.indexOf('_') !== -1) {
      return name.split('_').join(' ')
    } else {
      return name
    }
  },
  convertProductPrice(products) {
    for (const product of Array.from(products)) {
      if (!product.picture) {
        product.picture = []
      }

      product.description = product.description.replace(/\n/g, '<br>')
      const decimalNum = Array.from(ZERO_DECIMAL_CURRENCY_LIST).includes(
        __PRODUCT_NAME__ === 'sxl' ? 'CNY' : 'USD', // _settings.currencyCode in EcommerceStore
      )
        ? 0
        : 2

      product.variations = _.sortBy(product.variations, variant =>
        Number(variant.id),
      )

      for (const variant of Array.from(product.variations)) {
        /*
         * hack price adapting if iframe wrapped whole site,
         * productchange event will trigger twice,
         * and secound time will re-use prevState,
         * eg: input 99, will wrong final result 0.99
         */
        if (!/\.\d{2}$/.test(variant.price)) {
          variant.price = (variant.price / 100).toFixed(decimalNum)
        }
      }
    }
    return products
  },
  _formatCategoryOption(category) {
    let label = ''
    if (category.level === 1) {
      label = category.name
    } else if (category.level === 2) {
      label = `\u00A0\u00A0\u00A0\u00A0\u00A0${category.name}`
    }

    return { value: category.name, label, id: category.id }
  },
  formattedCategoryOptions(categories, formatter = this._formatCategoryOption) {
    const categoryOptions = []
    const rawCategories = categories.toJS ? categories.toJS() : categories
    rawCategories.forEach(category => {
      categoryOptions.push(formatter(category))
    })
    return categoryOptions
  },
  addLabelForCategory(categories) {
    const newCategories = categories.map(category => {
      let label
      if (!category.get('level') || category.get('level') === 1) {
        label =
          category.get('name') +
          (category.get('products_count') ? category.get('products_count') : '')
      } else if (category.get('level') === 2) {
        label = `\u00A0\u00A0\u00A0\u00A0\u00A0${category.get(
          'name',
        )} (${category.get('products_count')})`
      }
      return category.set('label', label)
    })
    return newCategories
  },
  sortedCategories(categoryObjs, orderList) {
    orderList = orderList || {}
    let allSortedCategory = []
    const sortedCategory = categoryObjs
      .filter(category => category.level === 1)
      .sort((a, b) => {
        const aIndex = orderList[a.id] || -a.id
        const bIndex = orderList[b.id] || -b.id
        return aIndex >= bIndex ? 1 : -1
      })

    sortedCategory.forEach(category => {
      const subCategories =
        (category.children &&
          category.children.map(subCategoryId =>
            categoryObjs.find(category => category.id === subCategoryId),
          )) ||
        []

      const sortedChildren = subCategories.sort((a, b) => {
        const aIndex =
          (category.children_category_order &&
            category.children_category_order[a.id]) ||
          -a.id
        const bIndex =
          (category.children_category_order &&
            category.children_category_order[b.id]) ||
          -b.id
        return aIndex >= bIndex ? 1 : -1
      })
      allSortedCategory.push([category, ...sortedChildren])
    })

    allSortedCategory = flatten(allSortedCategory)
    return allSortedCategory
  },
  isSubCategory(item) {
    const category = item.toJS ? item.toJS() : item
    if (!item) {
      return false
    }
    return (
      path(['payload', 'item', 'parent_id'])(category) &&
      pathEq(['payload', 'item', 'level'], 2)(category)
    )
  },
  sortedCategory(categories, orderList) {
    orderList = orderList || {}
    // param default value not accurate enough
    // sortedCategory(categories, orderList = {})  =>  var orderList = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {}; // not check `null` param case here
    const categoriesObj =
      categories && categories.toJS ? categories.toJS() : categories
    if (categoriesObj) {
      return categoriesObj.sort((a, b) => {
        const aIndex = orderList[a.id] || -a.id
        const bIndex = orderList[b.id] || -b.id
        return aIndex >= bIndex ? 1 : -1
      })
    }
  },
  checkHasSubCategory(categories) {
    return (
      categories &&
      categories.some(item => {
        if (categories.toJS) {
          return item.get('children') && item.get('children').size > 0
        } else {
          return item.children && item.children.length > 0
        }
      })
    )
  },
}
